<?php

namespace UrbanGoListing\CPT\Shortcodes\Listing;

use UrbanGoListing\Lib;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

class MapWithListings implements Lib\ShortcodeInterface {
	private $base;
	
	public function __construct() {
		$this->base = 'edgtf_map_with_listings';
		
		// Listing selected items filter
		add_filter( 'vc_autocomplete_edgtf_map_with_listings_selected_items_callback', array( &$this, 'listingIdAutocompleteSuggester', ), 10, 1 ); // Get suggestion(find). Must return an array
		
		// Listing selected items render
		add_filter( 'vc_autocomplete_edgtf_map_with_listings_selected_items_render', array( &$this, 'listingIdAutocompleteRender', ), 10, 1 ); // Render exact listing. Must return an array (label,value)
		
		add_action( 'vc_before_init', array( $this, 'vcMap' ) );
	}
	
	public function getBase() {
		return $this->base;
	}
	
	public function vcMap() {
		if ( function_exists( 'vc_map' ) ) {
			vc_map( array(
					'name'     => esc_html__( 'Map With Listings', 'urbango-listing' ),
					'base'     => $this->getBase(),
					'category' => esc_html__( 'by URBANGO LISTING', 'urbango-listing' ),
					'icon'     => 'icon-wpb-map-with-listings extended-custom-listing-icon',
					'params'   => array(
						array(
							'type'        => 'textfield',
							'param_name'  => 'map_height',
							'heading'     => esc_html__( 'Map Height', 'urbango-listing' ),
							'description' => esc_html__( 'Default value is 500px', 'urbango-listing' )
						),
						array(
							'type'        => 'textfield',
							'param_name'  => 'number_of_items',
							'heading'     => esc_html__( 'Number of Items', 'urbango-listing' ),
							'description' => esc_html__( 'Set number of items for your list. Enter -1 to show all.', 'urbango-listing' ),
							'value'       => '-1'
						),
						array(
							'type'        => 'autocomplete',
							'param_name'  => 'selected_items',
							'heading'     => esc_html__( 'Show Only Items with Listed IDs', 'urbango-listing' ),
							'settings'    => array(
								'multiple'      => true,
								'sortable'      => true,
								'unique_values' => true
							),
							'description' => esc_html__( 'Delimit ID numbers by comma (leave empty for all)', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'order_by',
							'heading'     => esc_html__( 'Order By', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_listing_list_query_order_by_array() )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'order',
							'heading'     => esc_html__( 'Order', 'urbango-listing' ),
							'value'       => array_flip( urbango_edge_get_query_order_array() )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'category',
							'heading'     => esc_html__( 'By Category', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_cpt_taxonomy_items( 'category' ) )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'location',
							'heading'     => esc_html__( 'By Location', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_cpt_taxonomy_items( 'location' ) )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'tag',
							'heading'     => esc_html__( 'By Tag', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_cpt_taxonomy_items( 'tag' ) )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'amenity',
							'heading'     => esc_html__( 'By Amenity', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_cpt_taxonomy_items( 'amenity' ) )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'author_id',
							'heading'     => esc_html__( 'Author', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_all_users() )
						)
					)
				)
			);
		}
	}
	
	public function render( $atts, $content = null ) {
		$args   = array(
			'map_height'      => '',
			'number_of_items' => '-1',
			'selected_items'  => '',
			'order_by'        => 'date',
			'order'           => 'DESC',
			'category'        => '',
			'location'        => '',
			'tag'             => '',
			'amenity'         => '',
			'author_id'       => ''
		);
		$params = shortcode_atts( $args, $atts );
		
		$params['map_styles'] = $this->getMapStyles( $params );
		$params['have_items'] = $this->checkIsItemsExist();
		$params['maps_args']  = $this->setMapArgs( $params );
		
		$html = urbango_listing_get_module_template_part( 'cpt-listing', 'map-with-listings', 'holder', '', $params, array(), false );
		
		return $html;
	}
	
	private function setMapArgs( $params ) {
		$args = array(
			'posts_per_page' => $params['number_of_items'],
			'order'          => $params['order']
		);
		
		$items_ids = null;
		if ( ! empty( $params['selected_items'] ) ) {
			$items_ids        = explode( ',', $params['selected_items'] );
			$args['post__in'] = $items_ids;
		}
		
		// Taxonomy query values
		if ( ! empty( $params['category'] ) || ! empty( $params['location'] ) || ! empty( $params['tag'] ) || ! empty( $params['amenity'] ) ) {
			$tax_query = array();
			
			if ( ! empty( $params['category'] ) ) {
				$tax_query[] = array(
					'taxonomy' => 'listing-category',
					'terms'    => $params['category']
				);
			}
			
			if ( ! empty( $params['location'] ) ) {
				$location = esc_attr( $params['location'] );
			
				if ( is_int( intval( $location ) ) && intval( $location ) > 0 ) {
					$tax_query[] = array(
						'taxonomy' => 'listing-location',
						'terms'    => $location
					);
				} else {
					$tax_query[] = array(
						'taxonomy' => 'listing-location',
						'field'    => 'slug',
						'terms'    => strtolower( $location ),
						'compare'  => 'LIKE'
					);
				}
			}
			
			if ( ! empty( $params['tag'] ) ) {
				$tax_query[] = array(
					'taxonomy' => 'listing-tag',
					'terms'    => $params['tag']
				);
			}
			
			if ( ! empty( $params['amenity'] ) ) {
				$tax_query[] = array(
					'taxonomy' => 'listing-amenity',
					'terms'    => $params['amenity']
				);
			}
			
			$args['tax_query'] = $tax_query;
		}
		
		// Custom meta query args
		$order_by   = ! empty( $params['order_by'] ) ? $params['order_by'] : 'date';
		$query_args = array( 'relation' => 'AND' );
		
		// Custom order meta query args
		if ( $order_by === 'featured-first' || $order_by === 'featured-last' ) {
			$args['orderby'] = 'edgtf_listing_is_featured_meta post_date';
			$args['order']   = $order_by === 'featured-first' ? 'DESC' : 'ASC';
			
			$query_args[] = array(
				'relation' => 'OR',
				array(
					'key'     => 'edgtf_listing_is_featured_meta',
					'value'   => array( 'yes', 'no', '' ),
					'compare' => 'IN'
				),
				array(
					'key'     => 'edgtf_listing_is_featured_meta',
					'compare' => 'NOT EXISTS'
				)
			);
		} else if ( $order_by === 'price-range-high' ) {
			$args['orderby'] = 'meta_value';
			
			$query_args[] = array(
				'key'     => 'edgtf_listing_single_price_range_meta',
				'value'   => '5',
				'compare' => '<='
			);
		} else if ( $order_by === 'price-range-low' ) {
			$args['orderby'] = 'meta_value';
			$args['order']   = 'ASC';
			
			$query_args[] = array(
				'key'     => 'edgtf_listing_single_price_range_meta',
				'value'   => '1',
				'compare' => '>='
			);
		} else {
			$args['orderby'] = $order_by;
		}
		
		// Author ID meta query args
		if ( ! empty( $params['author_id'] ) ) {
			$query_array['author'] = $params['author_id'];
		}
		
		$args['meta_query'] = $query_args;
		
		if ( ! empty( $params['next_page'] ) ) {
			$args['paged'] = $params['next_page'];
		} else {
			$args['paged'] = 1;
		}
		
		return $args;
	}
	
	private function getMapStyles( $params ) {
		$styles = array();
		
		if ( ! empty( $params['map_height'] ) ) {
			$styles[] = 'height: ' . urbango_edge_filter_px( $params['map_height'] ) . 'px';
		}
		
		return implode( ';', $styles );
	}
	
	private function checkIsItemsExist() {
		$items = urbango_listing_get_cpt_items( 'listing-item' );
		
		return ! empty( $items );
	}
	
	/**
	 * Filter listing items by ID or Title
	 *
	 * @param $query
	 *
	 * @return array
	 */
	public function listingIdAutocompleteSuggester( $query ) {
		global $wpdb;
		$listing_id      = (int) $query;
		$post_meta_infos = $wpdb->get_results( $wpdb->prepare( "SELECT ID AS id, post_title AS title
					FROM {$wpdb->posts} 
					WHERE post_type = 'listing-item' AND ( ID = '%d' OR post_title LIKE '%%%s%%' )", $listing_id > 0 ? $listing_id : - 1, stripslashes( $query ), stripslashes( $query ) ), ARRAY_A );
		
		$results = array();
		if ( is_array( $post_meta_infos ) && ! empty( $post_meta_infos ) ) {
			foreach ( $post_meta_infos as $value ) {
				$data          = array();
				$data['value'] = $value['id'];
				$data['label'] = esc_html__( 'Id', 'urbango-listing' ) . ': ' . $value['id'] . ( ( strlen( $value['title'] ) > 0 ) ? ' - ' . esc_html__( 'Title', 'urbango-listing' ) . ': ' . $value['title'] : '' );
				$results[]     = $data;
			}
		}
		
		return $results;
	}
	
	/**
	 * Find listing by id
	 * @since 4.4
	 *
	 * @param $query
	 *
	 * @return bool|array
	 */
	public function listingIdAutocompleteRender( $query ) {
		$query = trim( $query['value'] ); // get value from requested
		if ( ! empty( $query ) ) {
			// get listing
			$listing = get_post( (int) $query );
			if ( ! is_wp_error( $listing ) ) {
				
				$listing_id    = $listing->ID;
				$listing_title = $listing->post_title;
				
				$listing_title_display = '';
				if ( ! empty( $listing_title ) ) {
					$listing_title_display = ' - ' . esc_html__( 'Title', 'urbango-listing' ) . ': ' . $listing_title;
				}
				
				$listing_id_display = esc_html__( 'Id', 'urbango-listing' ) . ': ' . $listing_id;
				
				$data          = array();
				$data['value'] = $listing_id;
				$data['label'] = $listing_id_display . $listing_title_display;
				
				return ! empty( $data ) ? $data : false;
			}
			
			return false;
		}
		
		return false;
	}
}
