<?php

namespace UrbanGoListing\CPT\Shortcodes\Listing;

use UrbanGoListing\Lib;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

class LocationBanner implements Lib\ShortcodeInterface {
	private $base;
	
	public function __construct() {
		$this->base = 'edgtf_location_banner';
		
		add_action( 'vc_before_init', array( $this, 'vcMap' ) );
	}
	
	public function getBase() {
		return $this->base;
	}
	
	public function vcMap() {
		if ( function_exists( 'vc_map' ) ) {
			vc_map( array(
					'name'     => esc_html__( 'Location Banner', 'urbango-listing' ),
					'base'     => $this->getBase(),
					'category' => esc_html__( 'by URBANGO LISTING', 'urbango-listing' ),
					'icon'     => 'icon-wpb-location-banner extended-custom-listing-icon',
					'params'   => array(
						array(
							'type'       => 'dropdown',
							'param_name' => 'content_position',
							'heading'    => esc_html__( 'Content Info Position', 'urbango-listing' ),
							'value'      => array(
								esc_html__( 'On Image Bottom', 'urbango-listing' ) => 'bottom',
								esc_html__( 'On Image Middle', 'urbango-listing' ) => 'middle'
							)
						),
						array(
							'type'        => 'attach_image',
							'param_name'  => 'image',
							'heading'     => esc_html__( 'Custom Image', 'urbango-listing' ),
							'description' => esc_html__( 'Select image from media library or if you leave empty this option then default image will be location featured image', 'urbango-listing' )
						),
						array(
							'type'       => 'textfield',
							'param_name' => 'subtitle',
							'heading'    => esc_html__( 'Subtitle', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'location',
							'heading'     => esc_html__( 'Location', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_cpt_taxonomy_items( 'location' ) ),
							'save_always' => true
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'image_proportions',
							'heading'     => esc_html__( 'Image Proportions', 'urbango-listing' ),
							'value'       => array(
								esc_html__( 'Original', 'urbango-listing' )  => 'full',
								esc_html__( 'Square', 'urbango-listing' )    => 'square',
								esc_html__( 'Landscape', 'urbango-listing' ) => 'landscape',
								esc_html__( 'Portrait', 'urbango-listing' )  => 'portrait',
								esc_html__( 'Thumbnail', 'urbango-listing' ) => 'thumbnail',
								esc_html__( 'Medium', 'urbango-listing' )    => 'medium',
								esc_html__( 'Large', 'urbango-listing' )     => 'large'
							),
							'description' => esc_html__( 'Set image proportions for your list. Default value is full', 'urbango-listing' ),
							'group'       => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_title',
							'heading'    => esc_html__( 'Enable Title', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'title_tag',
							'heading'    => esc_html__( 'Title Tag', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_title_tag( true, array( 'p' => 'p' ) ) ),
							'dependency' => array( 'element' => 'enable_title', 'value' => array( 'yes' ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'colorpicker',
							'param_name' => 'title_color',
							'heading'    => esc_html__( 'Title Color', 'urbango-listing' ),
							'dependency' => array( 'element' => 'enable_title', 'value' => array( 'yes' ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'subtitle_tag',
							'heading'     => esc_html__( 'Subtitle Tag', 'urbango-listing' ),
							'value'       => array_flip( urbango_edge_get_title_tag( true, array( 'p' => 'p', 'span' => esc_html__( 'Custom Heading', 'urbango-listing' ) ) ) ),
							'save_always' => true,
							'dependency'  => array( 'element' => 'subtitle', 'not_empty' => true ),
							'group'       => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'colorpicker',
							'param_name' => 'subtitle_color',
							'heading'    => esc_html__( 'Subtitle Color', 'urbango-listing' ),
							'dependency' => array( 'element' => 'subtitle', 'not_empty' => true ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'textfield',
							'param_name' => 'subtitle_bottom_margin',
							'heading'    => esc_html__( 'Subtitle Bottom Margin (px)', 'urbango-listing' ),
							'dependency' => array( 'element' => 'title', 'not_empty' => true ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_items_count',
							'heading'    => esc_html__( 'Enable Items Count', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						)
					)
				)
			);
		}
	}
	
	public function render( $atts, $content = null ) {
		$args   = array(
			'content_position'       => 'bottom',
			'image'                  => '',
			'subtitle'               => '',
			'location'               => '',
			'image_proportions'      => 'full',
			'enable_title'           => 'yes',
			'title_tag'              => 'h2',
			'title_color'            => '',
			'subtitle_tag'           => 'span',
			'subtitle_color'         => '',
			'subtitle_bottom_margin' => '',
			'enable_items_count'     => 'yes'
		);
		$params = shortcode_atts( $args, $atts );
		
		/***
		 * @params query_results
		 */
		$additional_params = array();
		
		$additional_params['query_results']  = $this->getQueryArray( $params );
		$additional_params['holder_classes'] = $this->getHolderClasses( $params, $args );
		
		$params['title_tag']       = ! empty( $params['title_tag'] ) ? $params['title_tag'] : $args['title_tag'];
		$params['title_styles']    = $this->getTitleStyles( $params );
		$params['subtitle_tag']    = ! empty( $params['subtitle_tag'] ) ? $params['subtitle_tag'] : $args['subtitle_tag'];
		$params['subtitle_styles'] = $this->getSubtitleStyles( $params );
		
		$html = urbango_listing_get_module_template_part( 'cpt-listing', 'location-banner', 'holder', '', $params, $additional_params, false );
		
		return $html;
	}
	
	/**
	 * Generates query results
	 *
	 * @param $params
	 *
	 * @return array
	 */
	public function getQueryArray( $params ) {
		$query_args = array(
			'taxonomy' => 'listing-location',
			'number'   => 1
		);
		
		$location = trim( $params['location'] );
		if ( ! empty( $location ) ) {
			$query_args['include'] = array( intval( $location ) );
		}
		
		$query_array = get_terms( $query_args );
		
		return $query_array;
	}
	
	/**
	 * Generates holder classes
	 *
	 * @param $params
	 * @param $args
	 *
	 * @return string
	 */
	public function getHolderClasses( $params, $args ) {
		$classes = array();
		
		$classes[] = ! empty( $params['content_position'] ) ? 'edgtf-lb-info-' . $params['content_position'] : 'edgtf-lb-info-' . $args['content_position'];
		
		return implode( ' ', $classes );
	}
	
	/**
	 * Generates image size
	 *
	 * @param $params
	 *
	 * @return string
	 */
	public function getImageSize( $params ) {
		$thumb_size = 'full';
		
		if ( ! empty( $params['image_proportions'] ) ) {
			$image_size = $params['image_proportions'];
			
			switch ( $image_size ) {
				case 'landscape':
					$thumb_size = 'urbango_edge_image_landscape';
					break;
				case 'portrait':
					$thumb_size = 'urbango_edge_image_portrait';
					break;
				case 'square':
					$thumb_size = 'urbango_edge_image_square';
					break;
				case 'thumbnail':
					$thumb_size = 'thumbnail';
					break;
				case 'medium':
					$thumb_size = 'medium';
					break;
				case 'large':
					$thumb_size = 'large';
					break;
				case 'full':
					$thumb_size = 'full';
					break;
			}
		}
		
		return $thumb_size;
	}
	
	/**
	 * Generates title styles
	 *
	 * @param $params
	 *
	 * @return string
	 */
	private function getTitleStyles( $params ) {
		$styles = array();
		
		if ( ! empty( $params['title_color'] ) ) {
			$styles[] = 'color: ' . $params['title_color'];
		}
		
		return implode( ';', $styles );
	}
	
	/**
	 * Generates subtitle styles
	 *
	 * @param $params
	 *
	 * @return string
	 */
	private function getSubtitleStyles( $params ) {
		$styles = array();
		
		if ( ! empty( $params['subtitle_color'] ) ) {
			$styles[] = 'color: ' . $params['subtitle_color'];
		}
		
		if ( ! empty( $params['subtitle_bottom_margin'] ) ) {
			$styles[] = 'margin-bottom: ' . urbango_edge_filter_px( $params['subtitle_bottom_margin'] ) . 'px';
		}
		
		return implode( ';', $styles );
	}
}
