(function ($) {
	'use strict';
	
	var enquiry = {};
	edgtf.modules.enquiry = enquiry;
	
	enquiry.edgtfOnDocumentReady = edgtfOnDocumentReady;
	
	$(document).ready(edgtfOnDocumentReady);
	
	/*
	 All functions to be called on $(document).ready() should be in this function
	 */
	function edgtfOnDocumentReady() {
		edgtfInitEnquiryForm();
	}
	
	/**
	 * Initializes enquiry form
	 */
	function edgtfInitEnquiryForm() {
		var singleHolder = $('.edgtf-listing-single-holder'),
			enquiryHolder = $('.edgtf-enquiry-holder');
		
		if (enquiryHolder.length) {
			var openerButton = singleHolder.find('.edgtf-enquiry-opener'),
				enquiryForm = enquiryHolder.find('.edgtf-enquiry-form'),
				enquiryFormCloseButton = enquiryHolder.find('.edgtf-enquiry-close'),
				responseMessage = enquiryHolder.find('.edgtf-enquiry-response');
			
			openerButton.on('click', function () {
				enquiryHolder.fadeIn(300).addClass('edgtf-opened');
			});
			
			enquiryFormCloseButton.on('click', function (e) {
				e.preventDefault();
				
				enquiryHolder.fadeOut(300).removeClass('edgtf-opened');
			});
			
			// Close form on esc key
			$(window).on('keyup', function (e) {
				if (enquiryHolder.hasClass('edgtf-opened') && e.keyCode === 27) {
					enquiryHolder.fadeOut(300).removeClass('edgtf-opened');
				}
			});
			
			enquiryForm.on('submit', function (e) {
				e.preventDefault();
				
				responseMessage.empty().html('<span class="fa fa-spinner fa-spin" aria-hidden="true"></span>');
				
				var enquiryData = {
					name: enquiryForm.find('#edgtf-enquiry-name').val(),
					email: enquiryForm.find('#edgtf-enquiry-email').val(),
					message: enquiryForm.find('#edgtf-enquiry-message').val(),
					itemID: enquiryForm.find('#edgtf-enquiry-item-id').val(),
					nonce: enquiryForm.find('#urbango_listing_nonce_listing_item_enquiry_form').val()
				};
				
				var requestData = {
					action: 'urbango_listing_send_listing_item_enquiry',
					enquiryData: enquiryData
				};
				
				$.ajax({
					type: "POST",
					url: edgtfGlobalVars.vars.edgtfAjaxUrl,
					data: requestData,
					success: function (data) {
						var response = JSON.parse(data);
						
						if (response.status === 'success') {
							responseMessage.html(response.message);
							
							setTimeout(function () {
								enquiryHolder.fadeOut(300).removeClass('edgtf-opened');
							}, 800);
						} else {
							responseMessage.html(response.message);
						}
					}
				});
			});
		}
	}
	
})(jQuery);
(function ($) {
    "use strict";

    $( document ).on(
        'edgtfGoogleMapsCallbackEvent',
        function () {
            function CustomMarker( options ) {
                this.latlng = new google.maps.LatLng( {lat: options.position.lat, lng: options.position.lng} );
                this.setMap( options.map );
                this.templateData = options.templateData;
                this.markerData   = {
                    termIcon : options.termIcon
                };
            }

            CustomMarker.prototype = new google.maps.OverlayView();

            CustomMarker.prototype.draw = function () {
                var self = this;
                var div  = this.div;

                if ( ! div) {
                    div           = this.div = document.createElement( 'div' );
                    var id        = this.templateData.itemId;
                    div.className = 'edgtf-map-marker-holder';
                    div.setAttribute( "id", id );
                    div.setAttribute( "data-latlng", this.latlng );

                    var markerInfoTemplate = _.template( $( '.edgtf-info-window-template' ).html() );
                    markerInfoTemplate     = markerInfoTemplate( self.templateData );

                    var markerTemplate = _.template( $( '.edgtf-marker-template' ).html() );
                    markerTemplate     = markerTemplate( self.markerData );

                    $( div ).append( markerInfoTemplate );
                    $( div ).append( markerTemplate );

                    div.style.position = 'absolute';
                    div.style.cursor   = 'pointer';

                    var panes = this.getPanes();
                    panes.overlayImage.appendChild( div );
                }

                var point = this.getProjection().fromLatLngToDivPixel( this.latlng );

                if (point) {
                    div.style.left = (point.x) + 'px';
                    div.style.top  = (point.y) + 'px';
                }
            };

            CustomMarker.prototype.remove = function () {
                if (this.div) {
                    this.div.parentNode.removeChild( this.div );
                    this.div = null;
                }
            };

            CustomMarker.prototype.getPosition = function () {
                return this.latlng;
            };

            window.edgtfCustomMarker = CustomMarker;
        }
    );

})( jQuery );

(function ($) {
	"use strict";
	
	var listingMaps = {};
	edgtf.modules.listingMaps = listingMaps;
	listingMaps.edgtfInitMultipleListingMap = edgtfInitMultipleListingMap;
	listingMaps.edgtfInitMobileMap = edgtfInitMobileMap;
	listingMaps.edgtfReinitMultipleGoogleMaps = edgtfReinitMultipleGoogleMaps;
	listingMaps.edgtfGoogleMaps = {};
	
	$(window).on('load', edgtfOnWindowLoad);
	
	function edgtfOnWindowLoad() {
		edgtfInitMobileMap();
		edgtfBindListTitlesAndMap();
	}

	$( document ).on(
		'edgtfGoogleMapsCallbackEvent',
		function () {
			edgtfInitMultipleListingMap();
		}
	);
	
	function edgtfInitMultipleListingMap() {
		var mapHolder = $('#edgtf-listing-multiple-map-holder');
		
		if (mapHolder.length) {
			edgtf.modules.listingMaps.edgtfGoogleMaps.getDirectoryItemsAddresses({
				mapHolder: 'edgtf-listing-multiple-map-holder',
				hasFilter: true
			});
		}
	}
	
	function edgtfInitMobileMap() {
		var mapOpener = $('.edgtf-listing-view-larger-map a'),
			mapOpenerIcon = mapOpener.children('i'),
			mapHolder = $('.edgtf-map-holder');
		
		if (mapOpener.length) {
			mapOpener.on('click', function (e) {
				e.preventDefault();
				
				if (mapHolder.hasClass('edgtf-fullscreen-map')) {
					mapHolder.removeClass('edgtf-fullscreen-map');
					mapOpenerIcon.removeClass('icon-basic-magnifier-minus');
					mapOpenerIcon.addClass('icon-basic-magnifier-plus');
				} else {
					mapHolder.addClass('edgtf-fullscreen-map');
					mapOpenerIcon.removeClass('icon-basic-magnifier-plus');
					mapOpenerIcon.addClass('icon-basic-magnifier-minus');
				}
				
				edgtf.modules.listingMaps.edgtfGoogleMaps.getDirectoryItemsAddresses();
			});
		}
	}
	
	function edgtfReinitMultipleGoogleMaps(addresses, action) {
		if (action === 'append') {
			var mapObjs = edgtfMultipleMapVars.multiple.addresses.concat(addresses);
			edgtfMultipleMapVars.multiple.addresses = mapObjs;
			
			edgtf.modules.listingMaps.edgtfGoogleMaps.getDirectoryItemsAddresses({
				addresses: mapObjs
			});
		} else if (action === 'replace') {
			edgtfMultipleMapVars.multiple.addresses = addresses;
			edgtf.modules.listingMaps.edgtfGoogleMaps.getDirectoryItemsAddresses({
				addresses: addresses
			});
		}
		
		edgtfBindListTitlesAndMap();
	}
	
	function edgtfBindListTitlesAndMap() {
		var itemsList = $('.edgtf-map-list-holder');
		
		if (itemsList.length) {
			itemsList.each(function () {
				var thisItemsList = $(this),
					listItems = thisItemsList.find('article'),
					map = thisItemsList.find('.edgtf-map-list-map-part');
				
				if (map.length) {
					listItems.each(function () {
						//Init hover
						var listItem = $(this);
						
						if (!listItem.hasClass('edgtf-init')) {
							listItem.mouseenter(function () {
								var itemId = listItem.data('id'),
									inactiveMarkersHolder = $('.edgtf-map-marker-holder:not(.edgtf-map-active)'),
									clusterMarkersHolder = $('.edgtf-cluster-marker');
								
								if (inactiveMarkersHolder.length) {
									inactiveMarkersHolder.removeClass('edgtf-active');
									$('#' + itemId + '.edgtf-map-marker-holder').addClass('edgtf-active');
								}
								
								if (clusterMarkersHolder.length) {
									clusterMarkersHolder.each(function () {
										var thisClusterMarker = $(this),
											clusterMarkersItemIds = thisClusterMarker.data('item-ids');
									
										if (clusterMarkersItemIds !== undefined && clusterMarkersItemIds.includes(itemId.toString())) {
											thisClusterMarker.addClass('edgtf-active');
										}
									});
								}
							}).mouseleave(function () {
								var markersHolder = $('.edgtf-map-marker-holder'),
									clusterMarkersHolder = $('.edgtf-cluster-marker');
								
								if (markersHolder.length) {
									markersHolder.each(function () {
										var thisMapHolder = $(this);
										
										if (!thisMapHolder.hasClass('edgtf-map-active')) {
											thisMapHolder.removeClass('edgtf-active');
										}
									});
								}
								
								if (clusterMarkersHolder.length) {
									clusterMarkersHolder.removeClass('edgtf-active');
								}
							});
							
							listItem.addClass('edgtf-init');
						}
					});
				}
			});
		}
	}
	
	listingMaps.edgtfGoogleMaps = {
		//Object variables
		mapHolder: {},
		map: {},
		markers: {},
		radius: {},
		circle: {},
		
		/**
		 * Returns map with single address
		 *
		 * @param options
		 */
		getDirectoryItemAddress: function (options) {
			/**
			 * use edgtfMapsVars to get variables for address, latitude, longitude by default
			 */
			var defaults = {
				location: edgtfSingleMapVars.single['currentListing'].location,
				zoom: 16,
				mapHolder: '',
				draggable: edgtfMapsVars.global.draggable,
				mapTypeControl: edgtfMapsVars.global.mapTypeControl,
				scrollwheel: edgtfMapsVars.global.scrollable,
				streetViewControl: edgtfMapsVars.global.streetViewControl,
				zoomControl: edgtfMapsVars.global.zoomControl,
				title: edgtfSingleMapVars.single['currentListing'].title,
				itemId: edgtfSingleMapVars.single['currentListing'].itemId,
				content: '',
				styles: edgtfMapsVars.global.mapStyle,
				termIcon: edgtfSingleMapVars.single['currentListing'].termIcon,
				featuredImage: edgtfSingleMapVars.single['currentListing'].featuredImage,
				itemUrl: edgtfSingleMapVars.single['currentListing'].itemUrl
			};
			var settings = $.extend({}, defaults, options);
			
			//Save variables for later usage
			this.mapHolder = settings.mapHolder;
			
			//Get map holder
			var mapHolder = document.getElementById(settings.mapHolder);
			
			//Initialize map
			var map = new google.maps.Map(mapHolder, {
				zoom: settings.zoom,
				draggable: settings.draggable,
				mapTypeControl: settings.mapTypeControl,
				scrollwheel: settings.scrollwheel,
				streetViewControl: settings.streetViewControl,
				zoomControl: settings.zoomControl
			});
			
			//Set map style
			map.setOptions({
				styles: settings.styles
			});
			
			//Try to locate by latitude and longitude
			if (typeof settings.location !== 'undefined' && settings.location !== null) {
				var latLong = {
					lat: parseFloat(settings.location.latitude),
					lng: parseFloat(settings.location.longitude)
				};
				//Set map center to location
				map.setCenter(latLong);
				//Add marker to map
				
				var templateData = {
					title: settings.title,
					itemId: settings.itemId,
					address: settings.location.address,
					featuredImage: settings.featuredImage,
					itemUrl: settings.itemUrl
				};
				
				var customMarker = new window.edgtfCustomMarker({
					map: map,
					position: latLong,
					templateData: templateData,
					termIcon: settings.termIcon
				});
				
				this.initMarkerInfo();
			}
		},
		
		/**
		 * Returns map with multiple addresses
		 *
		 * @param options
		 */
		getDirectoryItemsAddresses: function (options) {
			var defaults = {
				geolocation: false,
				mapHolder: 'edgtf-listing-multiple-map-holder',
				addresses: edgtfMultipleMapVars.multiple.addresses,
				draggable: edgtfMapsVars.global.draggable,
				mapTypeControl: edgtfMapsVars.global.mapTypeControl,
				scrollwheel: edgtfMapsVars.global.scrollable,
				streetViewControl: edgtfMapsVars.global.streetViewControl,
				zoomControl: edgtfMapsVars.global.zoomControl,
				zoom: 16,
				styles: edgtfMapsVars.global.mapStyle,
				radius: 50, //radius for marker visibility, in km
				hasFilter: false
			};
			var settings = $.extend({}, defaults, options);
			
			//Get map holder
			var mapHolder = document.getElementById(settings.mapHolder);
			
			//Initialize map
			var map = new google.maps.Map(mapHolder, {
				zoom: settings.zoom,
				draggable: settings.draggable,
				mapTypeControl: settings.mapTypeControl,
				scrollwheel: settings.scrollwheel,
				streetViewControl: settings.streetViewControl,
				zoomControl: settings.zoomControl
			});
			
			//Save variables for later usage
			this.mapHolder = settings.mapHolder;
			this.map = map;
			this.radius = settings.radius;
			
			//Set map style
			map.setOptions({
				styles: settings.styles
			});
			
			//If geolocation enabled set map center to user location
			if (navigator.geolocation && settings.geolocation) {
				this.centerOnCurrentLocation();
			}
			
			//Filter addresses, remove items without latitude and longitude
			var addresses = [];
			
			if (typeof settings.addresses !== 'undefined') {
				var addressesLength = settings.addresses.length;
				
				if (settings.addresses.length !== null) {
					for (var i = 0; i < addressesLength; i++) {
						var location = settings.addresses[i].location;
						
						if (typeof location !== 'undefined' && location !== null) {
							
							if (location.latitude !== '' && location.longitude !== '') {
								addresses.push(settings.addresses[i]);
							}
						}
					}
				}
			}
			
			//Center map and set borders of map
			this.setMapBounds(addresses);
			
			//Add markers to the map
			this.addMultipleMarkers(addresses);
		},
		
		/**
		 * Add multiple markers to map
		 */
		addMultipleMarkers: function (markersData) {
			var map = this.map;
			var markers = [];
			
			//Loop through markers
			var len = markersData.length;
			
			for (var i = 0; i < len; i++) {
				var latLng = {
					lat: parseFloat(markersData[i].location.latitude),
					lng: parseFloat(markersData[i].location.longitude)
				};
				
				//Custom html markers
				//Insert marker data into info window template
				var templateData = {
					title: markersData[i].title,
					itemId: markersData[i].itemId,
					address: markersData[i].location.address,
					featuredImage: markersData[i].featuredImage,
					itemUrl: markersData[i].itemUrl,
					latLng: latLng
				};
				
				var customMarker = new window.edgtfCustomMarker({
					position: latLng,
					map: map,
					templateData: templateData,
					termIcon: markersData[i].termIcon
				});
				
				markers.push(customMarker);
			}
			
			this.markers = markers;
			
			//Init map clusters ( Grouping map markers at small zoom values )
			this.initMapClusters();
			
			//Init marker info
			this.initMarkerInfo();
		},
		
		/**
		 * Set map bounds for Map with multiple markers
		 *
		 * @param addressesArray
		 */
		setMapBounds: function (addressesArray) {
			var bounds = new google.maps.LatLngBounds();
			
			for (var i = 0; i < addressesArray.length; i++) {
				bounds.extend(new google.maps.LatLng(parseFloat(addressesArray[i].location.latitude), parseFloat(addressesArray[i].location.longitude)));
			}
			
			this.map.fitBounds(bounds);
		},
		
		/**
		 * Init map clusters for grouping markers on small zoom values
		 */
		initMapClusters: function () {
			
			//Activate clustering on multiple markers
			var markerClusteringOptions = {
				minimumClusterSize: 2,
				maxZoom: 12,
				styles: [{
					width: 50,
					height: 60,
					url: '',
					textSize: 12
				}]
			};
			
			var markerClusterer = new MarkerClusterer(this.map, this.markers, markerClusteringOptions);
		},
		
		initMarkerInfo: function () {
			var map = this.map;
			
			$(document).off('click', '.edgtf-map-marker').on('click', '.edgtf-map-marker', function (e) {
				var self = $(this),
					markerHolders = $('.edgtf-map-marker-holder'),
					infoWindows = $('.edgtf-info-window'),
					markerHolder = self.parent('.edgtf-map-marker-holder'),
					markerlatlngData = markerHolder.data('latlng'),
					infoWindow = self.siblings('.edgtf-info-window');
				
				if (markerHolder.hasClass('edgtf-active edgtf-map-active')) {
					markerHolder.removeClass('edgtf-active edgtf-map-active');
					infoWindow.fadeOut(0);
				} else {
					markerHolders.removeClass('edgtf-active edgtf-map-active');
					infoWindows.fadeOut(0);
					markerHolder.addClass('edgtf-active edgtf-map-active');
					infoWindow.fadeIn(300);
					
					if (markerlatlngData.length && markerlatlngData !== undefined) {
						var latlngStr = markerlatlngData.replace('(', '').replace(')', '').split(',', 2);
						var lat = parseFloat(latlngStr[0]);
						var lng = parseFloat(latlngStr[1]);
						
						map.setCenter(new google.maps.LatLng(lat, lng));
					}
				}
			});
		},
		
		/**
		 * If geolocation enabled center map on users current position
		 */
		centerOnCurrentLocation: function (setInputAddressValue, placesInput, geoLocationLinkIcon, listingListHolder) {
			var map = this.map;
			
			// Try HTML5 geolocation.
			if (navigator.geolocation) {
				if (setInputAddressValue) {
					geoLocationLinkIcon.addClass('fa-spinner fa-spin');
				}
				
				navigator.geolocation.getCurrentPosition(
					function (position) {
						var lat = position.coords.latitude,
							lng = position.coords.longitude,
							latlng = {
								lat: lat,
								lng: lng
							};
						
						if (setInputAddressValue) {
							var geocoder = new google.maps.Geocoder(),
								cityName = '',
								cityWithCountryName = '';
							
							geocoder.geocode({'latLng': new google.maps.LatLng(lat, lng)}, function (results, status) {
								if (status === google.maps.GeocoderStatus.OK && typeof results === 'object') {
									var resultsObject = results;
									
									for (var $i = 0; $i <= resultsObject.length; $i++) {
										var result = resultsObject[$i];
										
										if (typeof result === 'object' && result.types[0] === 'locality') {
											var currentAddress = result.address_components;
											
											cityName = currentAddress[0].long_name;
											
											for (var $j = 0; $j <= currentAddress.length; $j++) {
												if (typeof currentAddress[$j] === 'object' && currentAddress[$j].types[0] === 'country') {
													cityWithCountryName = cityName + ',' + currentAddress[$j].long_name;
												}
											}
										}
									}
							
									if (typeof cityName === 'string') {
										geoLocationLinkIcon.removeClass('fa-spinner fa-spin');
										
										if (typeof cityWithCountryName === 'string') {
											placesInput.val(cityWithCountryName);
										} else {
											placesInput.val(cityName);
										}
										
										// ReInit listing list and map
										if (listingListHolder) {
											var locationObject = [];
											
											locationObject.push(cityName);
											locationObject.push(latlng);
											locationObject.push(true);
											
											edgtf.modules.listingList.edgtfInitGeoLocationRangeSlider().showRangeSlider(latlng, true);
											edgtf.modules.listingList.edgtfInitListingListPagination().getMainPagFunction(listingListHolder, 1, true, locationObject);
										}
									}
								}
							});
						} else {
							map.setCenter(latlng);
						}
					}
				);
			}
		},
		
		/**
		 * Center map on forward location position
		 */
		centerOnForwardLocation: function (forwardLocation, markerEnabled, addressName) {
			var map = this.map;
			
			if (typeof forwardLocation === 'object') {
				
				if (markerEnabled) {
					var customMarker = new window.edgtfCustomMarker({
						map: map,
						position: forwardLocation,
						templateData: {
							title: 'Your location is here',
							itemId: 'edgtf-geo-location-marker',
							address: addressName,
							featuredImage: '',
							itemUrl: ''
						}
					});
					
					listingMaps.edgtfGoogleMaps.initMarkerInfo();
				}
				
				map.setZoom(12);
				map.setCenter(forwardLocation);
			}
		},
		
		/**
		 * Center map on forward address name location
		 */
		centerOnForwardAddressLocation: function (addressName) {
			
			if (typeof addressName === 'string' && typeof google === 'object') {
				var geocoder = new google.maps.Geocoder();
				
				geocoder.geocode({'address': addressName}, function (results, status) {
					if (status === google.maps.GeocoderStatus.OK && typeof results[0] === 'object') {
						listingMaps.edgtfGoogleMaps.centerOnForwardLocation(results[0].geometry.location);
					}
				});
			}
		},
		
		/**
		 * Set radius for current geo location location
		 */
		setGeoLocationRadius: function (forwardLocation, radius, isActive) {
			var map = this.map,
				circle = this.circle,
				markers = this.markers;
			
			if (typeof forwardLocation === 'object' && typeof google === 'object') {
				
				if (isActive) {
					circle.setMap(null);
				}
				
				this.circle = new google.maps.Circle({
					map: map,
					center: forwardLocation,
					radius: parseInt(radius, 10) * 1000, // 1000 change meters to kilometers
					strokeWeight: 0,
					fillColor: '#fc475f',
					fillOpacity: 0.15
				});
				
				var currentCircle = this.circle;
				
				var itemsInArea = [];
				$.each(markers, function(i,marker) {
					if (currentCircle.getBounds().contains(marker.latlng)) {
						itemsInArea.push(marker.templateData.itemId);
					}
				});
				
				edgtf.modules.listingList.edgtfInitGeoLocationRangeSlider().disableItemsOutOfRange(itemsInArea);
			}
		},
		
		/**
		 * Create autocomplete places for forward input field
		 */
		createAutocompletePlaces: function (placeInputID, listingListHolder) {
			
			if (typeof google === 'object' && typeof google.maps.places === 'object') {
				var autocompleteConfig = {
					types: ['(cities)']
				};
				
				var autocomplete = new google.maps.places.Autocomplete(placeInputID, autocompleteConfig);
				
				autocomplete.addListener('place_changed', function () {
					// Enable reset icon in field
					$(placeInputID).next().show();
					
					if (listingListHolder) {
						var place = autocomplete.getPlace();
						
						if (!place.geometry) {
							window.alert("No details available for input: '" + place.name + "'");
							return;
						}
						
						var locationObject = [];
						
						locationObject.push(place.address_components[0].short_name);
						locationObject.push(place.geometry.location);
						locationObject.push(false);
						
						// ReInit listing list and map
						edgtf.modules.listingList.edgtfInitGeoLocationRangeSlider().reset();
						edgtf.modules.listingList.edgtfInitListingListPagination().getMainPagFunction(listingListHolder, 1, true, locationObject);
					}
				});
			}
		}
	};
	
})(jQuery);
// ==ClosureCompiler==
// @compilation_level ADVANCED_OPTIMIZATIONS
// @externs_url http://closure-compiler.googlecode.com/svn/trunk/contrib/externs/maps/google_maps_api_v3_3.js
// ==/ClosureCompiler==

/**
 * @name MarkerClusterer for Google Maps v3
 * @version version 1.0.2
 * @author Luke Mahe
 * @fileoverview
 * The library creates and manages per-zoom-level clusters for large amounts of
 * markers.
 */

/**
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


/**
 * A Marker Clusterer that clusters markers.
 *
 * @param {google.maps.Map} map The Google map to attach to.
 * @param {Array.<google.maps.Marker>=} opt_markers Optional markers to add to
 *   the cluster.
 * @param {Object=} opt_options support the following options:
 *     'gridSize': (number) The grid size of a cluster in pixels.
 *     'maxZoom': (number) The maximum zoom level that a marker can be part of a
 *                cluster.
 *     'zoomOnClick': (boolean) Whether the default behaviour of clicking on a
 *                    cluster is to zoom into it.
 *     'imagePath': (string) The base URL where the images representing
 *                  clusters will be found. The full URL will be:
 *                  {imagePath}[1-5].{imageExtension}
 *                  Default: '../images/m'.
 *     'imageExtension': (string) The suffix for images URL representing
 *                       clusters will be found. See _imagePath_ for details.
 *                       Default: 'png'.
 *     'averageCenter': (boolean) Whether the center of each cluster should be
 *                      the average of all markers in the cluster.
 *     'minimumClusterSize': (number) The minimum number of markers to be in a
 *                           cluster before the markers are hidden and a count
 *                           is shown.
 *     'styles': (object) An object that has style properties:
 *       'url': (string) The image url.
 *       'height': (number) The image height.
 *       'width': (number) The image width.
 *       'anchor': (Array) The anchor position of the label text.
 *       'textColor': (string) The text color.
 *       'textSize': (number) The text size.
 *       'backgroundPosition': (string) The position of the backgound x, y.
 * @constructor
 * @extends google.maps.OverlayView
 */
function MarkerClusterer(map, opt_markers, opt_options) {
    // MarkerClusterer implements google.maps.OverlayView interface. We use the
    // extend function to extend MarkerClusterer with google.maps.OverlayView
    // because it might not always be available when the code is defined so we
    // look for it at the last possible moment. If it doesn't exist now then
    // there is no point going ahead :)
    this.extend(MarkerClusterer, google.maps.OverlayView);
    this.map_ = map;

    /**
     * @type {Array.<google.maps.Marker>}
     * @private
     */
    this.markers_ = [];

    /**
     *  @type {Array.<Cluster>}
     */
    this.clusters_ = [];

    this.sizes = [53, 56, 66, 78, 90];

    /**
     * @private
     */
    this.styles_ = [];

    /**
     * @type {boolean}
     * @private
     */
    this.ready_ = false;

    var options = opt_options || {};

    /**
     * @type {number}
     * @private
     */
    this.gridSize_ = options['gridSize'] || 60;

    /**
     * @private
     */
    this.minClusterSize_ = options['minimumClusterSize'] || 2;


    /**
     * @type {?number}
     * @private
     */
    this.maxZoom_ = options['maxZoom'] || null;

    this.styles_ = options['styles'] || [];

    /**
     * @type {string}
     * @private
     */
    this.imagePath_ = options['imagePath'] ||
        this.MARKER_CLUSTER_IMAGE_PATH_;

    /**
     * @type {string}
     * @private
     */
    this.imageExtension_ = options['imageExtension'] ||
        this.MARKER_CLUSTER_IMAGE_EXTENSION_;

    /**
     * @type {boolean}
     * @private
     */
    this.zoomOnClick_ = true;

    if (options['zoomOnClick'] != undefined) {
        this.zoomOnClick_ = options['zoomOnClick'];
    }

    /**
     * @type {boolean}
     * @private
     */
    this.averageCenter_ = false;

    if (options['averageCenter'] != undefined) {
        this.averageCenter_ = options['averageCenter'];
    }

    this.setupStyles_();

    this.setMap(map);

    /**
     * @type {number}
     * @private
     */
    this.prevZoom_ = this.map_.getZoom();

    // Add the map event listeners
    var that = this;
    google.maps.event.addListener(this.map_, 'zoom_changed', function () {
        // Determines map type and prevent illegal zoom levels
        var zoom = that.map_.getZoom();
        var minZoom = that.map_.minZoom || 0;
        var maxZoom = Math.min(that.map_.maxZoom || 100,
            that.map_.mapTypes[that.map_.getMapTypeId()].maxZoom);
        zoom = Math.min(Math.max(zoom, minZoom), maxZoom);

        if (that.prevZoom_ != zoom) {
            that.prevZoom_ = zoom;
            that.resetViewport();
        }
    });

    google.maps.event.addListener(this.map_, 'idle', function () {
        that.redraw();
    });

    // Finally, add the markers
    if (opt_markers && (opt_markers.length || Object.keys(opt_markers).length)) {
        this.addMarkers(opt_markers, false);
    }
}


/**
 * The marker cluster image path.
 *
 * @type {string}
 * @private
 */
MarkerClusterer.prototype.MARKER_CLUSTER_IMAGE_PATH_ = '../images/m';


/**
 * The marker cluster image path.
 *
 * @type {string}
 * @private
 */
MarkerClusterer.prototype.MARKER_CLUSTER_IMAGE_EXTENSION_ = 'png';


/**
 * Extends a objects prototype by anothers.
 *
 * @param {Object} obj1 The object to be extended.
 * @param {Object} obj2 The object to extend with.
 * @return {Object} The new extended object.
 * @ignore
 */
MarkerClusterer.prototype.extend = function (obj1, obj2) {
    return (function (object) {
        for (var property in object.prototype) {
            this.prototype[property] = object.prototype[property];
        }
        return this;
    }).apply(obj1, [obj2]);
};


/**
 * Implementaion of the interface method.
 * @ignore
 */
MarkerClusterer.prototype.onAdd = function () {
    this.setReady_(true);
};

/**
 * Implementaion of the interface method.
 * @ignore
 */
MarkerClusterer.prototype.draw = function () {
};

/**
 * Sets up the styles object.
 *
 * @private
 */
MarkerClusterer.prototype.setupStyles_ = function () {
    if (this.styles_.length) {
        return;
    }

    for (var i = 0, size; size = this.sizes[i]; i++) {
        this.styles_.push({
            url: this.imagePath_ + (i + 1) + '.' + this.imageExtension_,
            height: size,
            width: size
        });
    }
};

/**
 *  Fit the map to the bounds of the markers in the clusterer.
 */
MarkerClusterer.prototype.fitMapToMarkers = function () {
    var markers = this.getMarkers();
    var bounds = new google.maps.LatLngBounds();
    for (var i = 0, marker; marker = markers[i]; i++) {
        bounds.extend(marker.getPosition());
    }

    this.map_.fitBounds(bounds);
};


/**
 *  Sets the styles.
 *
 *  @param {Object} styles The style to set.
 */
MarkerClusterer.prototype.setStyles = function (styles) {
    this.styles_ = styles;
};


/**
 *  Gets the styles.
 *
 *  @return {Object} The styles object.
 */
MarkerClusterer.prototype.getStyles = function () {
    return this.styles_;
};


/**
 * Whether zoom on click is set.
 *
 * @return {boolean} True if zoomOnClick_ is set.
 */
MarkerClusterer.prototype.isZoomOnClick = function () {
    return this.zoomOnClick_;
};

/**
 * Whether average center is set.
 *
 * @return {boolean} True if averageCenter_ is set.
 */
MarkerClusterer.prototype.isAverageCenter = function () {
    return this.averageCenter_;
};


/**
 *  Returns the array of markers in the clusterer.
 *
 *  @return {Array.<google.maps.Marker>} The markers.
 */
MarkerClusterer.prototype.getMarkers = function () {
    return this.markers_;
};


/**
 *  Returns the number of markers in the clusterer
 *
 *  @return {Number} The number of markers.
 */
MarkerClusterer.prototype.getTotalMarkers = function () {
    return this.markers_.length;
};


/**
 *  Sets the max zoom for the clusterer.
 *
 *  @param {number} maxZoom The max zoom level.
 */
MarkerClusterer.prototype.setMaxZoom = function (maxZoom) {
    this.maxZoom_ = maxZoom;
};


/**
 *  Gets the max zoom for the clusterer.
 *
 *  @return {number} The max zoom level.
 */
MarkerClusterer.prototype.getMaxZoom = function () {
    return this.maxZoom_;
};


/**
 *  The function for calculating the cluster icon image.
 *
 *  @param {Array.<google.maps.Marker>} markers The markers in the clusterer.
 *  @param {number} numStyles The number of styles available.
 *  @return {Object} A object properties: 'text' (string) and 'index' (number).
 *  @private
 */
MarkerClusterer.prototype.calculator_ = function (markers, numStyles) {
    var index = 0;
    var count = markers.length;
    var dv = count;
    while (dv !== 0) {
        dv = parseInt(dv / 10, 10);
        index++;
    }

    index = Math.min(index, numStyles);
    return {
        text: count,
        index: index
    };
};


/**
 * Set the calculator function.
 *
 * @param {function(Array, number)} calculator The function to set as the
 *     calculator. The function should return a object properties:
 *     'text' (string) and 'index' (number).
 *
 */
MarkerClusterer.prototype.setCalculator = function (calculator) {
    this.calculator_ = calculator;
};


/**
 * Get the calculator function.
 *
 * @return {function(Array, number)} the calculator function.
 */
MarkerClusterer.prototype.getCalculator = function () {
    return this.calculator_;
};


/**
 * Add an array of markers to the clusterer.
 *
 * @param {Array.<google.maps.Marker>} markers The markers to add.
 * @param {boolean=} opt_nodraw Whether to redraw the clusters.
 */
MarkerClusterer.prototype.addMarkers = function (markers, opt_nodraw) {
    if (markers.length) {
        for (var i = 0, marker; marker = markers[i]; i++) {
            this.pushMarkerTo_(marker);
        }
    } else if (Object.keys(markers).length) {
        for (var marker in markers) {
            this.pushMarkerTo_(markers[marker]);
        }
    }
    if (!opt_nodraw) {
        this.redraw();
    }
};


/**
 * Pushes a marker to the clusterer.
 *
 * @param {google.maps.Marker} marker The marker to add.
 * @private
 */
MarkerClusterer.prototype.pushMarkerTo_ = function (marker) {
    marker.isAdded = false;
    if (marker['draggable']) {
        // If the marker is draggable add a listener so we update the clusters on
        // the drag end.
        var that = this;
        google.maps.event.addListener(marker, 'dragend', function () {
            marker.isAdded = false;
            that.repaint();
        });
    }
    this.markers_.push(marker);
};


/**
 * Adds a marker to the clusterer and redraws if needed.
 *
 * @param {google.maps.Marker} marker The marker to add.
 * @param {boolean=} opt_nodraw Whether to redraw the clusters.
 */
MarkerClusterer.prototype.addMarker = function (marker, opt_nodraw) {
    this.pushMarkerTo_(marker);
    if (!opt_nodraw) {
        this.redraw();
    }
};


/**
 * Removes a marker and returns true if removed, false if not
 *
 * @param {google.maps.Marker} marker The marker to remove
 * @return {boolean} Whether the marker was removed or not
 * @private
 */
MarkerClusterer.prototype.removeMarker_ = function (marker) {
    var index = -1;
    if (this.markers_.indexOf) {
        index = this.markers_.indexOf(marker);
    } else {
        for (var i = 0, m; m = this.markers_[i]; i++) {
            if (m == marker) {
                index = i;
                break;
            }
        }
    }

    if (index == -1) {
        // Marker is not in our list of markers.
        return false;
    }

    marker.setMap(null);

    this.markers_.splice(index, 1);

    return true;
};


/**
 * Remove a marker from the cluster.
 *
 * @param {google.maps.Marker} marker The marker to remove.
 * @param {boolean=} opt_nodraw Optional boolean to force no redraw.
 * @return {boolean} True if the marker was removed.
 */
MarkerClusterer.prototype.removeMarker = function (marker, opt_nodraw) {
    var removed = this.removeMarker_(marker);

    if (!opt_nodraw && removed) {
        this.resetViewport();
        this.redraw();
        return true;
    } else {
        return false;
    }
};


/**
 * Removes an array of markers from the cluster.
 *
 * @param {Array.<google.maps.Marker>} markers The markers to remove.
 * @param {boolean=} opt_nodraw Optional boolean to force no redraw.
 */
MarkerClusterer.prototype.removeMarkers = function (markers, opt_nodraw) {
    // create a local copy of markers if required
    // (removeMarker_ modifies the getMarkers() array in place)
    var markersCopy = markers === this.getMarkers() ? markers.slice() : markers;
    var removed = false;

    for (var i = 0, marker; marker = markersCopy[i]; i++) {
        var r = this.removeMarker_(marker);
        removed = removed || r;
    }

    if (!opt_nodraw && removed) {
        this.resetViewport();
        this.redraw();
        return true;
    }
};


/**
 * Sets the clusterer's ready state.
 *
 * @param {boolean} ready The state.
 * @private
 */
MarkerClusterer.prototype.setReady_ = function (ready) {
    if (!this.ready_) {
        this.ready_ = ready;
        this.createClusters_();
    }
};


/**
 * Returns the number of clusters in the clusterer.
 *
 * @return {number} The number of clusters.
 */
MarkerClusterer.prototype.getTotalClusters = function () {
    return this.clusters_.length;
};


/**
 * Returns the google map that the clusterer is associated with.
 *
 * @return {google.maps.Map} The map.
 */
MarkerClusterer.prototype.getMap = function () {
    return this.map_;
};


/**
 * Sets the google map that the clusterer is associated with.
 *
 * @param {google.maps.Map} map The map.
 */
MarkerClusterer.prototype.setMap = function (map) {
    this.map_ = map;
};


/**
 * Returns the size of the grid.
 *
 * @return {number} The grid size.
 */
MarkerClusterer.prototype.getGridSize = function () {
    return this.gridSize_;
};


/**
 * Sets the size of the grid.
 *
 * @param {number} size The grid size.
 */
MarkerClusterer.prototype.setGridSize = function (size) {
    this.gridSize_ = size;
};


/**
 * Returns the min cluster size.
 *
 * @return {number} The grid size.
 */
MarkerClusterer.prototype.getMinClusterSize = function () {
    return this.minClusterSize_;
};

/**
 * Sets the min cluster size.
 *
 * @param {number} size The grid size.
 */
MarkerClusterer.prototype.setMinClusterSize = function (size) {
    this.minClusterSize_ = size;
};


/**
 * Extends a bounds object by the grid size.
 *
 * @param {google.maps.LatLngBounds} bounds The bounds to extend.
 * @return {google.maps.LatLngBounds} The extended bounds.
 */
MarkerClusterer.prototype.getExtendedBounds = function (bounds) {
    var projection = this.getProjection();

    // Turn the bounds into latlng.
    var tr = new google.maps.LatLng(bounds.getNorthEast().lat(),
        bounds.getNorthEast().lng());
    var bl = new google.maps.LatLng(bounds.getSouthWest().lat(),
        bounds.getSouthWest().lng());

    // Convert the points to pixels and the extend out by the grid size.
    var trPix = projection.fromLatLngToDivPixel(tr);
    trPix.x += this.gridSize_;
    trPix.y -= this.gridSize_;

    var blPix = projection.fromLatLngToDivPixel(bl);
    blPix.x -= this.gridSize_;
    blPix.y += this.gridSize_;

    // Convert the pixel points back to LatLng
    var ne = projection.fromDivPixelToLatLng(trPix);
    var sw = projection.fromDivPixelToLatLng(blPix);

    // Extend the bounds to contain the new bounds.
    bounds.extend(ne);
    bounds.extend(sw);

    return bounds;
};


/**
 * Determins if a marker is contained in a bounds.
 *
 * @param {google.maps.Marker} marker The marker to check.
 * @param {google.maps.LatLngBounds} bounds The bounds to check against.
 * @return {boolean} True if the marker is in the bounds.
 * @private
 */
MarkerClusterer.prototype.isMarkerInBounds_ = function (marker, bounds) {
    return bounds.contains(marker.getPosition());
};


/**
 * Clears all clusters and markers from the clusterer.
 */
MarkerClusterer.prototype.clearMarkers = function () {
    this.resetViewport(true);

    // Set the markers a empty array.
    this.markers_ = [];
};


/**
 * Clears all existing clusters and recreates them.
 * @param {boolean} opt_hide To also hide the marker.
 */
MarkerClusterer.prototype.resetViewport = function (opt_hide) {
    // Remove all the clusters
    for (var i = 0, cluster; cluster = this.clusters_[i]; i++) {
        cluster.remove();
    }

    // Reset the markers to not be added and to be invisible.
    for (var i = 0, marker; marker = this.markers_[i]; i++) {
        marker.isAdded = false;
        if (opt_hide) {
            marker.setMap(null);
        }
    }

    this.clusters_ = [];
};

/**
 *
 */
MarkerClusterer.prototype.repaint = function () {
    var oldClusters = this.clusters_.slice();
    this.clusters_.length = 0;
    this.resetViewport();
    this.redraw();

    // Remove the old clusters.
    // Do it in a timeout so the other clusters have been drawn first.
    window.setTimeout(function () {
        for (var i = 0, cluster; cluster = oldClusters[i]; i++) {
            cluster.remove();
        }
    }, 0);
};


/**
 * Redraws the clusters.
 */
MarkerClusterer.prototype.redraw = function () {
    this.createClusters_();
};


/**
 * Calculates the distance between two latlng locations in km.
 * @see http://www.movable-type.co.uk/scripts/latlong.html
 *
 * @param {google.maps.LatLng} p1 The first lat lng point.
 * @param {google.maps.LatLng} p2 The second lat lng point.
 * @return {number} The distance between the two points in km.
 * @private
 */
MarkerClusterer.prototype.distanceBetweenPoints_ = function (p1, p2) {
    if (!p1 || !p2) {
        return 0;
    }

    var R = 6371; // Radius of the Earth in km
    var dLat = (p2.lat() - p1.lat()) * Math.PI / 180;
    var dLon = (p2.lng() - p1.lng()) * Math.PI / 180;
    var a = Math.sin(dLat / 2) * Math.sin(dLat / 2) +
        Math.cos(p1.lat() * Math.PI / 180) * Math.cos(p2.lat() * Math.PI / 180) *
        Math.sin(dLon / 2) * Math.sin(dLon / 2);
    var c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    var d = R * c;
    return d;
};


/**
 * Add a marker to a cluster, or creates a new cluster.
 *
 * @param {google.maps.Marker} marker The marker to add.
 * @private
 */
MarkerClusterer.prototype.addToClosestCluster_ = function (marker) {
    var distance = 40000; // Some large number
    var clusterToAddTo = null;
    var pos = marker.getPosition();
    for (var i = 0, cluster; cluster = this.clusters_[i]; i++) {
        var center = cluster.getCenter();
        if (center) {
            var d = this.distanceBetweenPoints_(center, marker.getPosition());
            if (d < distance) {
                distance = d;
                clusterToAddTo = cluster;
            }
        }
    }

    if (clusterToAddTo && clusterToAddTo.isMarkerInClusterBounds(marker)) {
        clusterToAddTo.addMarker(marker);
    } else {
        var cluster = new Cluster(this);
        cluster.addMarker(marker);
        this.clusters_.push(cluster);
    }
};


/**
 * Creates the clusters.
 *
 * @private
 */
MarkerClusterer.prototype.createClusters_ = function () {
    if (!this.ready_) {
        return;
    }

    // Get our current map view bounds.
    // Create a new bounds object so we don't affect the map.
    var mapBounds = new google.maps.LatLngBounds(this.map_.getBounds().getSouthWest(),
        this.map_.getBounds().getNorthEast());
    var bounds = this.getExtendedBounds(mapBounds);

    for (var i = 0, marker; marker = this.markers_[i]; i++) {
        if (!marker.isAdded && this.isMarkerInBounds_(marker, bounds)) {
            this.addToClosestCluster_(marker);
        }
    }
};


/**
 * A cluster that contains markers.
 *
 * @param {MarkerClusterer} markerClusterer The markerclusterer that this
 *     cluster is associated with.
 * @constructor
 * @ignore
 */
function Cluster(markerClusterer) {
    this.markerClusterer_ = markerClusterer;
    this.map_ = markerClusterer.getMap();
    this.gridSize_ = markerClusterer.getGridSize();
    this.minClusterSize_ = markerClusterer.getMinClusterSize();
    this.averageCenter_ = markerClusterer.isAverageCenter();
    this.center_ = null;
    this.markers_ = [];
    this.bounds_ = null;
    this.clusterIcon_ = new ClusterIcon(this, markerClusterer.getStyles(),
        markerClusterer.getGridSize());
}

/**
 * Determins if a marker is already added to the cluster.
 *
 * @param {google.maps.Marker} marker The marker to check.
 * @return {boolean} True if the marker is already added.
 */
Cluster.prototype.isMarkerAlreadyAdded = function (marker) {
    if (this.markers_.indexOf) {
        return this.markers_.indexOf(marker) != -1;
    } else {
        for (var i = 0, m; m = this.markers_[i]; i++) {
            if (m == marker) {
                return true;
            }
        }
    }
    return false;
};


/**
 * Add a marker the cluster.
 *
 * @param {google.maps.Marker} marker The marker to add.
 * @return {boolean} True if the marker was added.
 */
Cluster.prototype.addMarker = function (marker) {
    if (this.isMarkerAlreadyAdded(marker)) {
        return false;
    }

    if (!this.center_) {
        this.center_ = marker.getPosition();
        this.calculateBounds_();
    } else {
        if (this.averageCenter_) {
            var l = this.markers_.length + 1;
            var lat = (this.center_.lat() * (l - 1) + marker.getPosition().lat()) / l;
            var lng = (this.center_.lng() * (l - 1) + marker.getPosition().lng()) / l;
            this.center_ = new google.maps.LatLng(lat, lng);
            this.calculateBounds_();
        }
    }

    marker.isAdded = true;
    this.markers_.push(marker);

    var len = this.markers_.length;
    if (len < this.minClusterSize_ && marker.getMap() != this.map_) {
        // Min cluster size not reached so show the marker.
        marker.setMap(this.map_);
    }

    if (len == this.minClusterSize_) {
        // Hide the markers that were showing.
        for (var i = 0; i < len; i++) {
            this.markers_[i].setMap(null);
        }
    }

    if (len >= this.minClusterSize_) {
        marker.setMap(null);
    }

    this.updateIcon();
    return true;
};


/**
 * Returns the marker clusterer that the cluster is associated with.
 *
 * @return {MarkerClusterer} The associated marker clusterer.
 */
Cluster.prototype.getMarkerClusterer = function () {
    return this.markerClusterer_;
};


/**
 * Returns the bounds of the cluster.
 *
 * @return {google.maps.LatLngBounds} the cluster bounds.
 */
Cluster.prototype.getBounds = function () {
    var bounds = new google.maps.LatLngBounds(this.center_, this.center_);
    var markers = this.getMarkers();
    for (var i = 0, marker; marker = markers[i]; i++) {
        bounds.extend(marker.getPosition());
    }
    return bounds;
};


/**
 * Removes the cluster
 */
Cluster.prototype.remove = function () {
    this.clusterIcon_.remove();
    this.markers_.length = 0;
    delete this.markers_;
};


/**
 * Returns the number of markers in the cluster.
 *
 * @return {number} The number of markers in the cluster.
 */
Cluster.prototype.getSize = function () {
    return this.markers_.length;
};


/**
 * Returns a list of the markers in the cluster.
 *
 * @return {Array.<google.maps.Marker>} The markers in the cluster.
 */
Cluster.prototype.getMarkers = function () {
    return this.markers_;
};


/**
 * Returns the center of the cluster.
 *
 * @return {google.maps.LatLng} The cluster center.
 */
Cluster.prototype.getCenter = function () {
    return this.center_;
};


/**
 * Calculated the extended bounds of the cluster with the grid.
 *
 * @private
 */
Cluster.prototype.calculateBounds_ = function () {
    var bounds = new google.maps.LatLngBounds(this.center_, this.center_);
    this.bounds_ = this.markerClusterer_.getExtendedBounds(bounds);
};


/**
 * Determines if a marker lies in the clusters bounds.
 *
 * @param {google.maps.Marker} marker The marker to check.
 * @return {boolean} True if the marker lies in the bounds.
 */
Cluster.prototype.isMarkerInClusterBounds = function (marker) {
    return this.bounds_.contains(marker.getPosition());
};


/**
 * Returns the map that the cluster is associated with.
 *
 * @return {google.maps.Map} The map.
 */
Cluster.prototype.getMap = function () {
    return this.map_;
};


/**
 * Updates the cluster icon
 */
Cluster.prototype.updateIcon = function () {
    var zoom = this.map_.getZoom();
    var mz = this.markerClusterer_.getMaxZoom();

    if (mz && zoom > mz) {
        // The zoom is greater than our max zoom so show all the markers in cluster.
        for (var i = 0, marker; marker = this.markers_[i]; i++) {
            marker.setMap(this.map_);
        }
        return;
    }

    if (this.markers_.length < this.minClusterSize_) {
        // Min cluster size not yet reached.
        this.clusterIcon_.hide();
        return;
    }

    var numStyles = this.markerClusterer_.getStyles().length;
    var sums = this.markerClusterer_.getCalculator()(this.markers_, numStyles);
    this.clusterIcon_.setCenter(this.center_);
    this.clusterIcon_.setSums(sums);
    this.clusterIcon_.show();
};


/**
 * A cluster icon
 *
 * @param {Cluster} cluster The cluster to be associated with.
 * @param {Object} styles An object that has style properties:
 *     'url': (string) The image url.
 *     'height': (number) The image height.
 *     'width': (number) The image width.
 *     'anchor': (Array) The anchor position of the label text.
 *     'textColor': (string) The text color.
 *     'textSize': (number) The text size.
 *     'backgroundPosition: (string) The background postition x, y.
 * @param {number=} opt_padding Optional padding to apply to the cluster icon.
 * @constructor
 * @extends google.maps.OverlayView
 * @ignore
 */
function ClusterIcon(cluster, styles, opt_padding) {
    cluster.getMarkerClusterer().extend(ClusterIcon, google.maps.OverlayView);

    this.styles_ = styles;
    this.padding_ = opt_padding || 0;
    this.cluster_ = cluster;
    this.center_ = null;
    this.map_ = cluster.getMap();
    this.div_ = null;
    this.sums_ = null;
    this.visible_ = false;

    this.setMap(this.map_);
}


/**
 * Triggers the clusterclick event and zoom's if the option is set.
 */
ClusterIcon.prototype.triggerClusterClick = function () {
    var markerClusterer = this.cluster_.getMarkerClusterer();

    // Trigger the clusterclick event.
    google.maps.event.trigger(markerClusterer.map_, 'clusterclick', this.cluster_);

    if (markerClusterer.isZoomOnClick()) {
        // Zoom into the cluster.
        this.map_.fitBounds(this.cluster_.getBounds());
    }
};


/**
 * Adding the cluster icon to the dom.
 * @ignore
 */
ClusterIcon.prototype.onAdd = function () {
    this.div_ = document.createElement('DIV');
    this.div_.className = 'edgtf-cluster-marker';
    if (this.visible_) {
        var clusterItems = this.cluster_.markers_;
        var clusterItemsIDs = [];

        if (typeof clusterItems === 'object') {
            for (var $i = 0; $i < clusterItems.length; $i++){
                clusterItemsIDs.push(clusterItems[$i].templateData.itemId);
            }
        }

        this.div_.setAttribute('data-item-ids', clusterItemsIDs);

        var pos = this.getPosFromLatLng_(this.center_);
        this.div_.style.cssText = this.createCss(pos);
        this.div_.innerHTML = '<div class="edgtf-cluster-marker-inner">' +
            '<span class="edgtf-cluster-marker-number">' + this.sums_.text + '</span>' +
            '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="37.875px" height="50.75px" viewBox="0 0 37.875 50.75" enable-background="new 0 0 37.875 50.75" xml:space="preserve"><g><path fill="#EF4960" d="M0,18.938C0,29.396,17.746,50.75,18.938,50.75V0C8.479,0,0,8.479,0,18.938z"/><path fill="#DC4458" d="M37.875,18.938C37.875,8.479,29.396,0,18.938,0v50.75C20.129,50.75,37.875,29.396,37.875,18.938z"/></g><circle fill="#FFFFFF" cx="18.938" cy="19.188" r="14.813"/></svg>' +
            '</div>';
    }

    var panes = this.getPanes();
    panes.overlayMouseTarget.appendChild(this.div_);

    var that = this;

    this.div_.addEventListener(
        'click',
        function () {
            that.triggerClusterClick();
        }
    );
};


/**
 * Returns the position to place the div dending on the latlng.
 *
 * @param {google.maps.LatLng} latlng The position in latlng.
 * @return {google.maps.Point} The position in pixels.
 * @private
 */
ClusterIcon.prototype.getPosFromLatLng_ = function (latlng) {
    var pos = this.getProjection().fromLatLngToDivPixel(latlng);
    pos.x -= parseInt(this.width_ / 2, 10);
    pos.y -= parseInt(this.height_ / 2, 10);
    return pos;
};


/**
 * Draw the icon.
 * @ignore
 */
ClusterIcon.prototype.draw = function () {
    if (this.visible_) {
        var pos = this.getPosFromLatLng_(this.center_);
        this.div_.style.top = pos.y + 'px';
        this.div_.style.left = pos.x + 'px';
        this.div_.style.zIndex = google.maps.Marker.MAX_ZINDEX + 1;
    }
};


/**
 * Hide the icon.
 */
ClusterIcon.prototype.hide = function () {
    if (this.div_) {
        this.div_.style.display = 'none';
    }
    this.visible_ = false;
};


/**
 * Position and show the icon.
 */
ClusterIcon.prototype.show = function () {
    if (this.div_) {
        var pos = this.getPosFromLatLng_(this.center_);
        this.div_.style.cssText = this.createCss(pos);
        this.div_.style.display = '';
    }
    this.visible_ = true;
};


/**
 * Remove the icon from the map
 */
ClusterIcon.prototype.remove = function () {
    this.setMap(null);
};


/**
 * Implementation of the onRemove interface.
 * @ignore
 */
ClusterIcon.prototype.onRemove = function () {
    if (this.div_ && this.div_.parentNode) {
        this.hide();
        this.div_.parentNode.removeChild(this.div_);
        this.div_ = null;
    }
};


/**
 * Set the sums of the icon.
 *
 * @param {Object} sums The sums containing:
 *   'text': (string) The text to display in the icon.
 *   'index': (number) The style index of the icon.
 */
ClusterIcon.prototype.setSums = function (sums) {
    this.sums_ = sums;
    this.text_ = sums.text;
    this.index_ = sums.index;
    if (this.div_) {
        this.div_.innerHTML = sums.text;
    }

    this.useStyle();
};


/**
 * Sets the icon to the the styles.
 */
ClusterIcon.prototype.useStyle = function () {
    var index = Math.max(0, this.sums_.index - 1);
    index = Math.min(this.styles_.length - 1, index);
    var style = this.styles_[index];
    this.url_ = style['url'];
    this.height_ = style['height'];
    this.width_ = style['width'];
    this.textColor_ = style['textColor'];
    this.anchor_ = style['anchor'];
    this.textSize_ = style['textSize'];
    this.backgroundPosition_ = style['backgroundPosition'];
};


/**
 * Sets the center of the icon.
 *
 * @param {google.maps.LatLng} center The latlng to set as the center.
 */
ClusterIcon.prototype.setCenter = function (center) {
    this.center_ = center;
};


/**
 * Create the css text based on the position of the icon.
 *
 * @param {google.maps.Point} pos The position.
 * @return {string} The css style text.
 */
ClusterIcon.prototype.createCss = function (pos) {
    var style = [];
    style.push('background-image:url(' + this.url_ + ');');
    var backgroundPosition = this.backgroundPosition_ ? this.backgroundPosition_ : '0 0';
    style.push('background-position:' + backgroundPosition + ';');

    if (typeof this.anchor_ === 'object') {
        if (typeof this.anchor_[0] === 'number' && this.anchor_[0] > 0 &&
            this.anchor_[0] < this.height_) {
            style.push('height:' + (this.height_ - this.anchor_[0]) +
                'px; padding-top:' + this.anchor_[0] + 'px;');
        } else {
            style.push('height:' + this.height_ + 'px; line-height:' + this.height_ +
                'px;');
        }
        if (typeof this.anchor_[1] === 'number' && this.anchor_[1] > 0 &&
            this.anchor_[1] < this.width_) {
            style.push('width:' + (this.width_ - this.anchor_[1]) +
                'px; padding-left:' + this.anchor_[1] + 'px;');
        } else {
            style.push('width:' + this.width_ + 'px; text-align:center;');
        }
    } else {
        style.push('height:' + this.height_ + 'px; line-height:' +
            this.height_ + 'px; width:' + this.width_ + 'px; text-align:center;');
    }

    var txtColor = this.textColor_ ? this.textColor_ : 'black';
    var txtSize = this.textSize_ ? this.textSize_ : 11;

    style.push('cursor:pointer; top:' + pos.y + 'px; left:' +
        pos.x + 'px; color:' + txtColor + '; position:absolute; font-size:' +
        txtSize + 'px; font-family:Arial,sans-serif; font-weight:bold');
    return style.join('');
};


// Export Symbols for Closure
// If you are not going to compile with closure then you can remove the
// code below.
var window = window || {};
window['MarkerClusterer'] = MarkerClusterer;
MarkerClusterer.prototype['addMarker'] = MarkerClusterer.prototype.addMarker;
MarkerClusterer.prototype['addMarkers'] = MarkerClusterer.prototype.addMarkers;
MarkerClusterer.prototype['clearMarkers'] =
    MarkerClusterer.prototype.clearMarkers;
MarkerClusterer.prototype['fitMapToMarkers'] =
    MarkerClusterer.prototype.fitMapToMarkers;
MarkerClusterer.prototype['getCalculator'] =
    MarkerClusterer.prototype.getCalculator;
MarkerClusterer.prototype['getGridSize'] =
    MarkerClusterer.prototype.getGridSize;
MarkerClusterer.prototype['getExtendedBounds'] =
    MarkerClusterer.prototype.getExtendedBounds;
MarkerClusterer.prototype['getMap'] = MarkerClusterer.prototype.getMap;
MarkerClusterer.prototype['getMarkers'] = MarkerClusterer.prototype.getMarkers;
MarkerClusterer.prototype['getMaxZoom'] = MarkerClusterer.prototype.getMaxZoom;
MarkerClusterer.prototype['getStyles'] = MarkerClusterer.prototype.getStyles;
MarkerClusterer.prototype['getTotalClusters'] =
    MarkerClusterer.prototype.getTotalClusters;
MarkerClusterer.prototype['getTotalMarkers'] =
    MarkerClusterer.prototype.getTotalMarkers;
MarkerClusterer.prototype['redraw'] = MarkerClusterer.prototype.redraw;
MarkerClusterer.prototype['removeMarker'] =
    MarkerClusterer.prototype.removeMarker;
MarkerClusterer.prototype['removeMarkers'] =
    MarkerClusterer.prototype.removeMarkers;
MarkerClusterer.prototype['resetViewport'] =
    MarkerClusterer.prototype.resetViewport;
MarkerClusterer.prototype['repaint'] =
    MarkerClusterer.prototype.repaint;
MarkerClusterer.prototype['setCalculator'] =
    MarkerClusterer.prototype.setCalculator;
MarkerClusterer.prototype['setGridSize'] =
    MarkerClusterer.prototype.setGridSize;
MarkerClusterer.prototype['setMaxZoom'] =
    MarkerClusterer.prototype.setMaxZoom;
MarkerClusterer.prototype['onAdd'] = MarkerClusterer.prototype.onAdd;
MarkerClusterer.prototype['draw'] = MarkerClusterer.prototype.draw;

Cluster.prototype['getCenter'] = Cluster.prototype.getCenter;
Cluster.prototype['getSize'] = Cluster.prototype.getSize;
Cluster.prototype['getMarkers'] = Cluster.prototype.getMarkers;

ClusterIcon.prototype['onAdd'] = ClusterIcon.prototype.onAdd;
ClusterIcon.prototype['draw'] = ClusterIcon.prototype.draw;
ClusterIcon.prototype['onRemove'] = ClusterIcon.prototype.onRemove;

Object.keys = Object.keys || function (o) {
    var result = [];
    for (var name in o) {
        if (o.hasOwnProperty(name))
            result.push(name);
    }
    return result;
};

if (typeof module === 'object') {
    module.exports = MarkerClusterer;
}
(function ($) {
	'use strict';
	
	var roles = {};
	edgtf.modules.roles = roles;
	
	roles.edgtfOnDocumentReady = edgtfOnDocumentReady;
	
	$(document).ready(edgtfOnDocumentReady);
	
	/*
	 All functions to be called on $(document).ready() should be in this function
	 */
	function edgtfOnDocumentReady() {
		edgtfInitRegisterSelect();
	}
	
	function edgtfInitRegisterSelect() {
		var holder = $('.edgtf-register-content-inner select');
		
		if (holder.length) {
			holder.each(function () {
				var thisHolder = $(this);
				
				thisHolder.select2({
					minimumResultsForSearch: Infinity
				});
			});
		}
	}
	
})(jQuery);
(function ($) {
	'use strict';
	
	var wishlist = {};
	edgtf.modules.wishlist = wishlist;
	
	wishlist.edgtfOnDocumentReady = edgtfOnDocumentReady;
	
	$(document).ready(edgtfOnDocumentReady);
	
	/*
	 All functions to be called on $(document).ready() should be in this function
	 */
	function edgtfOnDocumentReady() {
		edgtfInitWishlist();
	}
	
	/**
	 * Initializes wishlist functionality
	 */
	function edgtfInitWishlist() {
		var wishlistLink = $('.edgtf-wishlist-holder .edgtf-wishlist-link');
		
		if (wishlistLink.length) {
			wishlistLink.each(function(){
				var thisWishlistLink = $(this),
					wishlistIconHTML = thisWishlistLink.html(),
					responseMessage = thisWishlistLink.siblings('.edgtf-wishlist-response');
			
				thisWishlistLink.off().on('click', function (e) {
					e.preventDefault();
					
					if (edgtf.body.hasClass('logged-in')) {
						var itemID = thisWishlistLink.data('id');
						
						if (itemID !== 'undefined' && !thisWishlistLink.hasClass('edgtf-added')) {
							thisWishlistLink.html('<span class="fa fa-spinner fa-spin" aria-hidden="true"></span>');
							
							$.ajax({
								type: "POST",
								url: edgtfGlobalVars.vars.edgtfAjaxUrl,
								data: {
									action: 'urbango_listing_wishlist_ajax_functionality',
									wishlistData: {
										type: 'add',
										itemID: itemID
									}
								},
								success: function (data) {
									var response = JSON.parse(data);
									
									if (response.status === 'success') {
										thisWishlistLink.addClass('edgtf-added');
										responseMessage.html(response.message).addClass('edgtf-show').fadeIn(200);
										
										$(document).trigger('urbango_listing_wishlist_item_is_added', [itemID]);
									} else {
										responseMessage.html(response.message).addClass('edgtf-show').fadeIn(200);
									}
									
									setTimeout(function () {
										thisWishlistLink.html(wishlistIconHTML);
										
										var wishlistTitle = thisWishlistLink.find('.edgtf-wishlist-title');
										
										if (wishlistTitle.length) {
											wishlistTitle.text(wishlistTitle.data('added-title'));
										}
										
										responseMessage.fadeOut(300).removeClass('edgtf-show').empty();
									}, 800);
								}
							});
						}
					} else {
						// Trigger event.
						$(document.body).trigger('open_user_login_trigger');
					}
				});
			});
		}
	}
	
	$(document).on('urbango_listing_wishlist_item_is_removed', function (e, removedItemID) {
		var wishlistLink = $('.edgtf-wishlist-holder .edgtf-wishlist-link');
		
		if (wishlistLink.length) {
			wishlistLink.each(function(){
				var thisWishlistLink = $(this),
					wishlistTitle = thisWishlistLink.find('.edgtf-wishlist-title');
			
				if (thisWishlistLink.data('id') === removedItemID && thisWishlistLink.hasClass('edgtf-added')) {
					thisWishlistLink.removeClass('edgtf-added');
				
					if (wishlistTitle.length) {
						wishlistTitle.text(wishlistTitle.data('title'));
					}
				}
			});
		}
	});
	
})(jQuery);
(function ($) {
	'use strict';
	
	var claim = {};
	edgtf.modules.claim = claim;
	
	claim.edgtfOnDocumentReady = edgtfOnDocumentReady;
	
	$(document).ready(edgtfOnDocumentReady);
	
	/*
	 All functions to be called on $(document).ready() should be in this function
	 */
	function edgtfOnDocumentReady() {
		edgtfInitClaim();
	}
	
	/**
	 * Initializes claim functionality
	 */
	function edgtfInitClaim() {
		var claimOpener = $('.edgtf-claim-opener'),
			claimHolder = $('.edgtf-claim-form-holder');
		
		if (claimOpener.length) {
			
			if (claimHolder.length) {
				var claimForm = claimHolder.find('.edgtf-claim-form'),
					claimFormCloseButton = claimHolder.find('.edgtf-claim-close'),
					responseMessage = claimHolder.find('.edgtf-claim-form-response');
				
				claimFormCloseButton.on('click', function (e) {
					e.preventDefault();
					
					claimHolder.fadeOut(300).removeClass('edgtf-opened');
				});
				
				// Close form on esc key
				$(window).on('keyup', function (e) {
					if (claimHolder.hasClass('edgtf-opened') && e.keyCode === 27) {
						claimHolder.fadeOut(300).removeClass('edgtf-opened');
					}
				});
				
				claimForm.on('submit', function (e) {
					e.preventDefault();
					
					if (edgtf.body.hasClass('logged-in')) {
						responseMessage.empty().html('<span class="fa fa-spinner fa-spin" aria-hidden="true"></span>');
						
						var claimData = {
							message: claimForm.find('#edgtf-claim-message').val(),
							itemID: claimForm.data('id'),
							userID: claimForm.find('#edgtf-claim-user-id').val(),
							nonce: claimForm.find('#urbango_listing_nonce_listing_item_claim_form').val()
						};
						
						var requestData = {
							action: 'urbango_listing_send_listing_item_claim',
							claimData: claimData
						};
						
						$.ajax({
							type: "POST",
							url: edgtfGlobalVars.vars.edgtfAjaxUrl,
							data: requestData,
							success: function (data) {
								var response = JSON.parse(data);
								
								if (response.status === 'success') {
									responseMessage.html(response.message);
									
									setTimeout(function () {
										claimHolder.fadeOut(300).removeClass('edgtf-opened');
									}, 800);
								} else {
									responseMessage.html(response.message);
								}
							}
						});
					} else {
						// Trigger event.
						$(document.body).trigger('open_user_login_trigger');
					}
				});
			}
			
			claimOpener.each(function () {
				var thisClaimOpener = $(this),
					itemID = thisClaimOpener.data('id');
				
				thisClaimOpener.off().on('click', function (e) {
					e.preventDefault();
					
					if (edgtf.body.hasClass('logged-in')) {
						claimHolder.fadeIn(300).addClass('edgtf-opened');
						
						// Set default values
						claimForm.data('id', itemID).find('#edgtf-claim-message').val('');
						responseMessage.empty();
					} else {
						// Trigger event.
						$(document.body).trigger('open_user_login_trigger');
					}
				});
			});
		}
	}
	
})(jQuery);
(function ($) {
	'use strict';
	
	var listing = {};
	edgtf.modules.listing = listing;
	
	listing.edgtfOnDocumentReady = edgtfOnDocumentReady;
	
	$(document).ready(edgtfOnDocumentReady);
	
	/*
	 All functions to be called on $(document).ready() should be in this function
	 */
	function edgtfOnDocumentReady() {
		edgtfDeleteListingItemFromDashboard();
	}
	
	function edgtfDeleteListingItemFromDashboard() {
		var button = $('.edgtf-listing-item-delete');
		
		if (button.length) {
			button.each(function () {
				var thisDeleteButton = $(this),
					listingId = thisDeleteButton.data('listing-id'),
					confirmText = thisDeleteButton.data('confirm-text'),
					listing = thisDeleteButton.parents('.edgtf-lp-listing-item');
				
				thisDeleteButton.on('click', function (e) {
					e.preventDefault();
					
					var confirmDelete = confirm(confirmText);
					
					if (confirmDelete) {
						var dataForm = {
							'action': 'urbango_listing_delete_listing',
							'listing_id': listingId
						};
						
						$.ajax({
							type: 'POST',
							data: dataForm,
							url: edgtfGlobalVars.vars.edgtfAjaxUrl,
							success: function (data) {
								var response = JSON.parse(data);
								
								if (response.status === 'success') {
									listing.fadeOut(function () {
										listing.remove();
									});
								}
							}
						});
					}
				});
			});
		}
	}
	
})(jQuery);
(function ($) {
	'use strict';
	
	var report = {};
	edgtf.modules.report = report;
	
	report.edgtfOnDocumentReady = edgtfOnDocumentReady;
	
	$(document).ready(edgtfOnDocumentReady);
	
	/*
	 All functions to be called on $(document).ready() should be in this function
	 */
	function edgtfOnDocumentReady() {
		edgtfInitReport();
		edgtfResubmitReportedItemFromDashboard();
	}
	
	/**
	 * Initializes report functionality
	 */
	function edgtfInitReport() {
		var reportOpener = $('.edgtf-report-opener'),
			reportHolder = $('.edgtf-report-form-holder');
		
		if (reportOpener.length) {
			
			if (reportHolder.length) {
				var reportForm = reportHolder.find('.edgtf-report-form'),
					reportFormCloseButton = reportHolder.find('.edgtf-report-close'),
					responseMessage = reportHolder.find('.edgtf-report-form-response');
				
				reportFormCloseButton.on('click', function (e) {
					e.preventDefault();
					
					reportHolder.fadeOut(300).removeClass('edgtf-opened');
				});
				
				// Close form on esc key
				$(window).on('keyup', function (e) {
					if (reportHolder.hasClass('edgtf-opened') && e.keyCode === 27) {
						reportHolder.fadeOut(300).removeClass('edgtf-opened');
					}
				});
				
				reportForm.on('submit', function (e) {
					e.preventDefault();
					
					if (edgtf.body.hasClass('logged-in')) {
						responseMessage.empty().html('<span class="fa fa-spinner fa-spin" aria-hidden="true"></span>');
						
						var reportData = {
							message: reportForm.find('#edgtf-report-message').val(),
							itemID: reportForm.data('id'),
							userID: reportForm.find('#edgtf-report-user-id').val(),
							nonce: reportForm.find('#urbango_listing_nonce_listing_item_report_form').val()
						};
						
						var requestData = {
							action: 'urbango_listing_send_listing_item_report',
							reportData: reportData
						};
						
						$.ajax({
							type: "POST",
							url: edgtfGlobalVars.vars.edgtfAjaxUrl,
							data: requestData,
							success: function (data) {
								var response = JSON.parse(data);
								
								if (response.status === 'success') {
									responseMessage.html(response.message);
									
									setTimeout(function () {
										reportHolder.fadeOut(300).removeClass('edgtf-opened');
									}, 800);
								} else {
									responseMessage.html(response.message);
								}
							}
						});
					} else {
						// Trigger event.
						$(document.body).trigger('open_user_login_trigger');
					}
				});
			}
			
			reportOpener.each(function () {
				var thisReportOpener = $(this),
					itemID = thisReportOpener.data('id');
				
				thisReportOpener.off().on('click', function (e) {
					e.preventDefault();
					
					if (edgtf.body.hasClass('logged-in')) {
						reportHolder.fadeIn(300).addClass('edgtf-opened');
						
						// Set default values
						reportForm.data('id', itemID).find('#edgtf-report-message').val('');
						responseMessage.empty();
					} else {
						// Trigger event.
						$(document.body).trigger('open_user_login_trigger');
					}
				});
			});
		}
	}
	
	function edgtfResubmitReportedItemFromDashboard() {
		var button = $('.edgtf-listing-item-resubmit');
		
		if (button.length) {
			button.each(function () {
				var thisButton = $(this),
					listingId = thisButton.data('listing-id'),
					itemHolder = thisButton.parents('.edgtf-lp-reported-item'),
					messageHolder = itemHolder.find('.edgtf-resubmit-message'),
					responseHolder = itemHolder.find('.edgtf-lp-ri-response'),
					confirmText = thisButton.data('confirm-text');
				
				thisButton.on('click', function (e) {
					e.preventDefault();
					
					var confirmDelete = confirm(confirmText);

					if (confirmDelete) {
						thisButton.parent().append('<span class="edgtf-resubmit-spinner fa fa-spinner fa-spin" aria-hidden="true"></span>');

						var ajaxData = {
							'action': 'urbango_listing_resubmit_reported_item',
							'reportData': {
								itemID: listingId,
								message: messageHolder.val(),
								nonce: itemHolder.find('#urbango_listing_nonce_resubmit_reported_item').val()
							}
						};

						$.ajax({
							type: 'POST',
							data: ajaxData,
							url: edgtfGlobalVars.vars.edgtfAjaxUrl,
							success: function (data) {
								var response = JSON.parse(data);
								
								responseHolder.html(response.message);
								
								// Reset values
								thisButton.parent().find('.edgtf-resubmit-spinner').remove();
								messageHolder.val('');
								setTimeout(function(){
									responseHolder.empty();
								}, 3000);
							}
						});
					}
				});
			});
		}
	}
	
})(jQuery);
(function ($) {
	'use strict';
	
	var listingProfile = {};
	edgtf.modules.listingProfile = listingProfile;
	
	listingProfile.edgtfOnDocumentReady = edgtfOnDocumentReady;
	
	$(document).ready(edgtfOnDocumentReady);
	
	/*
	 All functions to be called on $(document).ready() should be in this function
	 */
	function edgtfOnDocumentReady() {
		edgtfInitSavedSearchesRemove();
	}
	
	function edgtfInitSavedSearchesRemove() {
		var searchesTab = $('.edgtf-listing-profile-searches');
		
		if (searchesTab.length) {
			var removeQueryButton = searchesTab.find('.edgtf-undo-query-save');
			
			removeQueryButton.on('click', function (e) {
				e.preventDefault();
				
				var thisButton = $(this);
				
				if (!confirm('Are you sure you want to remove this search?')) {
					return;
				}
				
				thisButton.html('<span class="fa fa-spinner fa-spin" aria-hidden="true"></span>');
				
				var ajaxData = {
					action: 'urbango_listing_set_listing_list_ajax_remove_query',
					query_id: thisButton.data('query-id')
				};
				
				$.ajax({
					type: 'POST',
					data: ajaxData,
					url: edgtfGlobalVars.vars.edgtfAjaxUrl,
					success: function (data) {
						var response = JSON.parse(data);
						
						if (response.status === 'success') {
							thisButton.parents('.edgtf-lp-ss-content-row').remove();
						} else {
							thisButton.html('<i class="fa fa-times" aria-hidden="true"></i>');
						}
					}
				});
			});
		}
	}
	
})(jQuery);
(function ($) {
	'use strict';
	
	var wishlistDropdown = {};
	edgtf.modules.wishlist = wishlistDropdown;
	
	wishlistDropdown.edgtfOnDocumentReady = edgtfOnDocumentReady;
	
	$(document).ready(edgtfOnDocumentReady);
	
	/*
	 All functions to be called on $(document).ready() should be in this function
	 */
	function edgtfOnDocumentReady() {
		edgtfInitWishlistDropdown();
	}
	
	/**
	 * Initializes wishlist dropdown functionality
	 */
	function edgtfInitWishlistDropdown() {
		var holder = $('.edgtf-wishlist-dropdown-holder');
		
		if (holder.length) {
			holder.each(function () {
				var thisHolder = $(this),
					link = thisHolder.find('.edgtf-wd-link'),
					numberOfItemsValue = link.find('.edgtf-wd-number-of-items'),
					itemsHolder = thisHolder.find('.edgtf-wd-items'),
					removeLink = itemsHolder.find('.edgtf-wd-remove-item');
				
				link.on('click', function (e) {
					e.preventDefault();
				});
				
				removeLink.off().on('click', function (e) {
					e.preventDefault();
					
					var thisRemoveLink = $(this),
						removeLinkHTML = thisRemoveLink.html(),
						removeItemID = thisRemoveLink.data('id');
					
					thisRemoveLink.html('<span class="fa fa-spinner fa-spin" aria-hidden="true"></span>');
					
					$.ajax({
						type: "POST",
						url: edgtfGlobalVars.vars.edgtfAjaxUrl,
						data: {
							action: 'urbango_listing_wishlist_ajax_functionality',
							wishlistData: {
								type: 'remove',
								itemID: removeItemID
							}
						},
						success: function (data) {
							var response = JSON.parse(data);
							
							if (response.status === 'success') {
								var newNumberOfItemsValue = parseInt(response.data['count'], 10);
						
								numberOfItemsValue.html(newNumberOfItemsValue);
								
								if (newNumberOfItemsValue === 0) {
									thisHolder.removeClass('edgtf-wd-has-items').addClass('edgtf-wd-no-items');
								}
								
								thisRemoveLink.closest('.edgtf-wd-item').fadeOut(200).remove();
								
								$(document).trigger('urbango_listing_wishlist_item_is_removed', [removeItemID]);
							} else {
								thisRemoveLink.html(removeLinkHTML);
							}
						}
					});
				});
			});
		}
	}
	
	$(document).on('urbango_listing_wishlist_item_is_added', function (e, addedItemID) {
		var holder = $('.edgtf-wishlist-dropdown-holder');
		
		if (holder.length) {
			holder.each(function () {
				var thisHolder = $(this),
					link = thisHolder.find('.edgtf-wd-link'),
					numberOfItemsValue = link.find('.edgtf-wd-number-of-items'),
					itemsHolder = thisHolder.find('.edgtf-wd-items');
				
				$.ajax({
					type: "POST",
					url: edgtfGlobalVars.vars.edgtfAjaxUrl,
					data: {
						action: 'urbango_listing_update_wishlist_dropdown_widget',
						wishlistData: {
							itemID: addedItemID
						}
					},
					success: function (data) {
						var response = JSON.parse(data);
						
						if (response.status === 'success') {
							numberOfItemsValue.html(parseInt(response.data['count'], 10));
							
							if (thisHolder.hasClass('edgtf-wd-no-items')) {
								thisHolder.removeClass('edgtf-wd-no-items').addClass('edgtf-wd-has-items');
							}
							
							itemsHolder.append(response.data['new_html']);
						}
					},
					complete: function () {
						edgtfInitWishlistDropdown();
					}
				});
			});
		}
	});
	
})(jQuery);
(function ($) {
	'use strict';
	
	var categoryTabs = {};
	edgtf.modules.categoryTabs = categoryTabs;
	
	categoryTabs.edgtfOnDocumentReady = edgtfOnDocumentReady;
	
	$(document).ready(edgtfOnDocumentReady);
	
	/*
	 All functions to be called on $(document).ready() should be in this function
	 */
	function edgtfOnDocumentReady() {
		initCategoryTabsSlider();
	}
	
	function initCategoryTabsSlider() {
		var holder = $('.edgtf-category-tabs-holder');
		
		if (holder.length && edgtf.windowWidth <= 680) {
			holder.each(function () {
				var thisHolder = $(this),
					sliderHolder = thisHolder.children(),
					numberOfItems = thisHolder.find('.edgtf-ct-item').length;
				
				if (numberOfItems > 5) {
					thisHolder.addClass('edgtf-slider-init');
					
					sliderHolder.addClass('edgtf-owl-slider').owlCarousel({
						items: 3,
						loop: true,
						autoplay: true,
						autoplayTimeout: 4000,
						smartSpeed: 800,
						dots: false,
						nav: false,
						onInitialize: function () {
							thisHolder.addClass('edgtf-slider-loaded');
						}
					});
				}
			});
		}
	}
	
})(jQuery);
(function ($) {
	'use strict';
	
	var listingList = {};
	edgtf.modules.listingList = listingList;
	listingList.edgtfInitGeoLocationRangeSlider = edgtfInitGeoLocationRangeSlider;
	listingList.edgtfInitListingListPagination = edgtfInitListingListPagination;
	
	listingList.edgtfOnDocumentReady = edgtfOnDocumentReady;
	listingList.edgtfOnWindowLoad = edgtfOnWindowLoad;
	listingList.edgtfOnWindowScroll = edgtfOnWindowScroll;
	
	$(document).ready(edgtfOnDocumentReady);
	$(window).on('load', edgtfOnWindowLoad);
	$(window).scroll(edgtfOnWindowScroll);
	
	/*
	 All functions to be called on $(document).ready() should be in this function
	 */
	function edgtfOnDocumentReady() {
		edgtfInitGeoLocationRangeSlider().init();
		edgtfInitListingListMapSwitcher();
	}
	
	/*
	 All functions to be called on $(window).on('load',) should be in this function
	 */
	function edgtfOnWindowLoad() {
		edgtfInitListingListFilter();
		edgtfInitListingListRelationTermsSwitcher();
		edgtfInitListingListPagination().init();
		edgtfListHoverClass();
	}
	
	/*
	 All functions to be called on $(window).scroll() should be in this function
	 */
	function edgtfOnWindowScroll() {
		edgtfInitListingListPagination().scroll();
	}
	
	/**
	 * Initializes listing list geo location radius functionality
	 */
	function edgtfInitGeoLocationRangeSlider() {
		var holder = $('.edgtf-listing-list-holder.edgtf-ll-with-map.edgtf-ll-with-filter');
		
		var createSlider = function (geoLocationRadius, slider) {
			noUiSlider.create(slider, {
				connect: [true, false],
				start: 0,
				tooltips: true,
				format: {
					from: function (value) {
						return parseFloat(value).toFixed(2);
					},
					to: function (value) {
						return parseFloat(value).toFixed(2);
					}
				},
				range: {
					min: 0,
					max: 50
				}
			});
			
			updateMapRadius(geoLocationRadius, slider);
		};
		
		var updateMapRadius = function (geoLocationRadius, slider) {
			var sliderEventCount = 0;
			
			slider.noUiSlider.on('set', function (values) {
				var geoLocation = geoLocationRadius.data('geo-location');
				
				if (typeof geoLocation === 'object') {
					edgtf.modules.listingMaps.edgtfGoogleMaps.setGeoLocationRadius(geoLocation, values, sliderEventCount > 0);
					sliderEventCount++;
				}
			});
		};
		
		var resetRadius = function (slider) {
			slider.noUiSlider.reset();
		};
		
		var sliderVisibility = function (geoLocationRadius, latlng, visibility) {
			geoLocationRadius.data('geo-location', latlng);
			
			if (visibility) {
				geoLocationRadius.show();
			} else {
				geoLocationRadius.hide();
			}
		};
		
		var disableItemsOutOfRange = function (itemsInArea) {
			var items = holder.find('.edgtf-ll-inner article');
			
			if (items.length && itemsInArea.length > 0) {
				if (!holder.children('.edgtf-ll-out-of-range-holder').length) {
					holder.append('<div class="edgtf-ll-out-of-range-holder"></div>');
				}
				
				var outOfRangeHolder = holder.children('.edgtf-ll-out-of-range-holder'),
					outOfRangeItems = outOfRangeHolder.children('article'),
					inRangeHolder = holder.find('.edgtf-ll-inner');
				
				items.each(function () {
					var thisItem = $(this),
						itemID = thisItem.data('id');
					
					if (itemID !== undefined && itemID !== false) {
						var itemInRange = false;
						
						$.each(itemsInArea, function (i, id) {
							if (parseInt(itemID, 10) === id) {
								itemInRange = true;
								return true;
							}
						});
						
						if (!itemInRange) {
							thisItem.appendTo(outOfRangeHolder);
							
							if (holder.hasClass('edgtf-ll-masonry')) {
								inRangeHolder.isotope('layout');
							}
						}
					}
				});
				
				if (outOfRangeItems.length) {
					outOfRangeItems.each(function () {
						var thisOutItem = $(this),
							outItemID = thisOutItem.data('id'),
							itemInRange = false;
						
						$.each(itemsInArea, function (i, id) {
							if (parseInt(outItemID, 10) === id) {
								itemInRange = true;
								return true;
							}
						});
						
						if (itemInRange) {
							thisOutItem.appendTo(inRangeHolder);
							
							if (holder.hasClass('edgtf-ll-masonry')) {
								inRangeHolder.isotope('layout');
							}
						}
					});
				}
			}
		};
		
		return {
			init: function () {
				if (holder.length) {
					var geoLocationRadius = holder.find('.edgtf-fs-places-geo-radius');
					
					if (geoLocationRadius.length) {
						var slider = document.getElementById('edgtf-range-slider-id');
						
						createSlider(geoLocationRadius, slider);
					}
				}
			},
			reset: function () {
				if (holder.length) {
					var geoLocationRadius = holder.find('.edgtf-fs-places-geo-radius');
					
					if (geoLocationRadius.length && geoLocationRadius.is(':visible')) {
						var slider = document.getElementById('edgtf-range-slider-id');
						
						sliderVisibility(geoLocationRadius, '', false);
						resetRadius(slider);
					}
				}
			},
			showRangeSlider: function (latlng, visibility) {
				if (holder.length) {
					var geoLocationRadius = holder.find('.edgtf-fs-places-geo-radius');
					
					if (geoLocationRadius.length) {
						sliderVisibility(geoLocationRadius, latlng, visibility);
					}
				}
			},
			disableItemsOutOfRange: function (itemsInArea) {
				if (holder.length && typeof itemsInArea === 'object') {
					disableItemsOutOfRange(itemsInArea);
				}
			}
		};
	}
	
	/**
	 * Initializes listing list map switcher functionality
	 */
	function edgtfInitListingListMapSwitcher() {
		var holder = $('.edgtf-listing-list-holder.edgtf-ll-with-map .edgtf-map-switcher');
		
		if (holder.length) {
			holder.each(function () {
				var thisHolder = $(this),
					mainHolder = thisHolder.parents('.edgtf-listing-list-holder'),
					fullMapSwitcher = thisHolder.children('.edgtf-map-switcher-full-map'),
					fullListSwitcher = thisHolder.children('.edgtf-map-switcher-full-list'),
					resetSwitcher = thisHolder.children('.edgtf-map-switcher-reset');
				
				fullMapSwitcher.on('click', function(e){
					e.preventDefault();
					
					fullMapSwitcher.hide();
					fullListSwitcher.show();
					resetSwitcher.show();
					mainHolder.removeClass('edgtf-switch-full-list').addClass('edgtf-switch-full-map');
					edgtf.modules.common.edgtfDisableScroll();
				});
				
				fullListSwitcher.on('click', function(e){
					e.preventDefault();
					
					fullMapSwitcher.show();
					fullListSwitcher.hide();
					resetSwitcher.show();
					mainHolder.removeClass('edgtf-switch-full-map').addClass('edgtf-switch-full-list');
					edgtf.modules.common.edgtfEnableScroll();
				});
				
				resetSwitcher.on('click', function(e){
					e.preventDefault();
					
					fullMapSwitcher.show();
					fullListSwitcher.show();
					resetSwitcher.hide();
					mainHolder.removeClass('edgtf-switch-full-map edgtf-switch-full-list');
					edgtf.modules.common.edgtfEnableScroll();
				});
			});
		}
	}
	
	/**
	 * Initializes filter functionality
	 */
	function edgtfInitListingListFilter() {
		var holder = $('.edgtf-ll-filter-holder');
		
		if (holder.length) {
			holder.each(function () {
				var thisHolder = $(this),
					thisListingList = thisHolder.parents('.edgtf-listing-list-holder'),
					defaultItemLayoutData = thisListingList.data('item-layout'),
					defaultNumberOfColumnsData = thisListingList.data('number-of-columns'),
					searchButton = thisHolder.find('.edgtf-ll-filter-search'),
					saveButton = thisHolder.find('.edgtf-ll-filter-save'),
					resetButton = thisHolder.find('.edgtf-ll-filter-reset'),
					resultsMessage = thisHolder.find('.edgtf-filter-query-results');
				
				// Filter by attributes
				var	select2Fields = thisHolder.find('.edgtf-fs-inner.edgtf-fs-is-select2'),
					select2FieldsExists = select2Fields.length,
					checkboxFields = thisHolder.find('.edgtf-fs-inner.edgtf-fs-is-checkbox'),
					checkboxFieldsExists = checkboxFields.length,
					inputTextFields = thisHolder.find('.edgtf-fs-inner.edgtf-fs-is-text'),
					inputTextFieldsExists = inputTextFields.length,
					switchLayout = thisHolder.find('.edgtf-filter-section-switch-layout'),
					switchLayoutIsSet = switchLayout.length,
					inputPlacesFields = thisHolder.find('.edgtf-fs-inner.edgtf-fs-is-places'),
					inputPlacesFieldsExists = inputPlacesFields.length;
				
				if (select2FieldsExists) {
					select2Fields.each(function () {
						var select2Item = $(this);
						
						initSelect2Field(select2Item, select2Item.data('type'));
					});
				}
				
				// Init search functionality
				searchButton.on('click', function (e) {
					e.preventDefault();
					
					if (select2FieldsExists) {
						select2Fields.each(function () {
							var select2Item = $(this),
								select2Type = select2Item.data('type');
							
							thisListingList.data(select2Type, select2Item.data(select2Type));
						});
					}
					
					if (checkboxFieldsExists) {
						checkboxFields.each(function () {
							var checkboxItem = $(this),
								checkedCBItems = [];
							
							checkboxItem.find('input[type="checkbox"]:checked').each(function () {
								checkedCBItems.push(parseInt($(this).data('id')));
							});
							
							if (checkedCBItems.length) {
								checkedCBItems = checkedCBItems.join(',');
							}
							
							thisListingList.data(checkboxItem.data('type'), checkedCBItems);
						});
					}
					
					if (inputTextFieldsExists) {
						inputTextFields.each(function () {
							var inputTextItem = $(this),
								inputTextValue = inputTextItem.find('input[type="text"]').val();
							
							thisListingList.data(inputTextItem.data('type'), inputTextValue);
						});
					}
					
					if (inputPlacesFieldsExists) {
						inputPlacesFields.each(function () {
							var inputPlaceItem = $(this),
								inputPlaceValue = inputPlaceItem.find('input[type="text"]').val();
							
							thisListingList.data(inputPlaceItem.data('type'), inputPlaceValue);
						});
					}
					
					edgtfInitListingListPagination().getMainPagFunction(thisListingList, 1, true);
					
					resetButton.addClass('edgtf-ll-filter-is-searched');
				});
				
				// Init save functionality
				saveButton.on('click', function (e) {
					e.preventDefault();
					
					if (edgtf.body.hasClass('logged-in')) {
						resultsMessage.html('<span class="fa fa-spinner fa-spin" aria-hidden="true"></span>');
						
						var ajaxData = {
							action: 'urbango_listing_set_listing_list_ajax_save_query'
						};
						
						if (select2FieldsExists) {
							select2Fields.each(function () {
								var select2Item = $(this),
									select2Type = select2Item.data('type');
								
								ajaxData[select2Type] = select2Item.data(select2Type);
							});
						}
						
						if (checkboxFieldsExists) {
							checkboxFields.each(function () {
								var checkboxItem = $(this),
									checkedCBItems = [];
								
								checkboxItem.find('input[type="checkbox"]:checked').each(function () {
									checkedCBItems.push(parseInt($(this).data('id')));
								});
								
								if (checkedCBItems.length) {
									checkedCBItems = checkedCBItems.join(',');
									
									ajaxData[checkboxItem.data('type')] = checkedCBItems;
								}
							});
						}
						
						if (inputTextFieldsExists) {
							inputTextFields.each(function () {
								var inputTextItem = $(this);
								
								ajaxData[inputTextItem.data('type')] = inputTextItem.find('input[type="text"]').val();
							});
						}
						
						if (inputPlacesFieldsExists) {
							inputPlacesFields.each(function () {
								var inputPlaceItem = $(this);
								
								ajaxData[inputPlaceItem.data('type')] = inputPlaceItem.find('input[type="text"]').val();
							});
						}
						
						$.ajax({
							type: 'POST',
							data: ajaxData,
							url: edgtfGlobalVars.vars.edgtfAjaxUrl,
							success: function (data) {
								var response = JSON.parse(data);

								resultsMessage.html('<span class="edgtf-filter-result-message">' + response.message + '</span>');
								resultsMessage.append(response.data);

								edgtfInitUndoQueryButton(resultsMessage.parent());
							}
						});
					} else {
						// Trigger event.
						$(document.body).trigger('open_user_login_trigger');
					}
				});
				
				// Init reset functionality
				resetButton.on('click', function (e) {
					e.preventDefault();
					
					if (select2FieldsExists) {
						select2Fields.each(function () {
							var select2Item = $(this),
								select2Type = select2Item.data('type');
							
							resetSelect2Field(select2Item, select2Type, select2Item.data('default-' + select2Type));
						});
					}
					
					if (checkboxFieldsExists) {
						checkboxFields.each(function () {
							$(this).find('input[type="checkbox"]').each(function () {
								$(this).prop('checked', false);
							});
						});
					}
					
					if (inputTextFieldsExists) {
						inputTextFields.each(function () {
							$(this).find('input[type="text"]').val('');
						});
					}
					
					if (inputPlacesFieldsExists) {
						inputPlacesFields.each(function () {
							$(this).find('input[type="text"]').val('');
						});
					}
					
					if (typeof edgtfInitListingListRelationTermsSwitcher === 'function') {
						edgtfInitListingListRelationTermsSwitcher(true);
					}
					
					if (resetButton.hasClass('edgtf-ll-filter-is-searched')) {
						resetButton.removeClass('edgtf-ll-filter-is-searched');
						
						searchButton.trigger('click');
					}
				});
				
				// Init switch layout functionality
				if (switchLayoutIsSet) {
					var standardLayoutLink = switchLayout.find('.edgtf-fs-sl-standard'),
						simpleLayoutLink = switchLayout.find('.edgtf-fs-sl-simple');
					
					if (defaultItemLayoutData.indexOf('standard') !== -1) {
						standardLayoutLink.addClass('edgtf-active');
					} else if (defaultItemLayoutData.indexOf('simple') !== -1) {
						simpleLayoutLink.addClass('edgtf-active');
					}
					
					standardLayoutLink.on('click', function (e) {
						e.preventDefault();
						
						var currentItemLayoutData = thisListingList.data('item-layout'),
							currentNumberOfColumnsData = thisListingList.data('number-of-columns');
						
						if (currentItemLayoutData !== 'standard') {
							standardLayoutLink.addClass('edgtf-active');
							simpleLayoutLink.removeClass('edgtf-active');
							
							thisListingList.data('item-layout', 'standard');
							thisListingList.data('number-of-columns', defaultNumberOfColumnsData);
							
							thisListingList.removeClass('edgtf-ll-layout-' + currentItemLayoutData + ' edgtf-' + currentNumberOfColumnsData + '-columns').addClass('edgtf-ll-layout-standard edgtf-' + defaultNumberOfColumnsData + '-columns');
						}
					});
					
					simpleLayoutLink.on('click', function (e) {
						e.preventDefault();
						
						var currentItemLayoutData = thisListingList.data('item-layout'),
							currentNumberOfColumnsData = thisListingList.data('number-of-columns');
						
						if (currentItemLayoutData !== 'simple') {
							simpleLayoutLink.addClass('edgtf-active');
							standardLayoutLink.removeClass('edgtf-active');
							
							thisListingList.data('item-layout', 'simple');
							thisListingList.data('number-of-columns', '1');
							
							thisListingList.removeClass('edgtf-ll-layout-' + currentItemLayoutData + ' edgtf-' + currentNumberOfColumnsData + '-columns').addClass('edgtf-ll-layout-simple edgtf-one-columns');
						}
					});
				}
				
				// Init geolocation and places functionality for location
				if(inputPlacesFieldsExists) {
					var placesInput = inputPlacesFields.find('input[type="text"]'),
						placesInputValue = placesInput.val(),
						placesInputID = document.getElementById('edgtf-fs-places-location'),
						geoLocationResetLink = inputPlacesFields.find('.edgtf-fs-places-reset'),
						geoLocationLink = inputPlacesFields.find('.edgtf-fs-places-geo'),
						geoLocationLinkIcon = geoLocationLink.children('.edgtf-fs-places-icon');
					
					if (placesInput.length) {
						edgtf.modules.listingMaps.edgtfGoogleMaps.createAutocompletePlaces(placesInputID, thisListingList);
					}
					
					if (placesInputValue.length && thisListingList.hasClass('edgtf-ll-with-map')) {
						geoLocationResetLink.show();
						edgtf.modules.listingMaps.edgtfGoogleMaps.centerOnForwardAddressLocation(placesInputValue);
					}
					
					geoLocationResetLink.on('click', function (e) {
						e.preventDefault();
						
						placesInput.val('');
						geoLocationResetLink.hide();
						
						edgtfInitGeoLocationRangeSlider().reset();
					});
					
					geoLocationLink.on('click', function (e) {
						e.preventDefault();
						
						if (typeof google === 'object' && typeof navigator === 'object') {
							geoLocationResetLink.show();
							edgtf.modules.listingMaps.edgtfGoogleMaps.centerOnCurrentLocation(true, placesInput, geoLocationLinkIcon, thisListingList);
						}
					});
				}
			});
		}
		
		function initSelect2Field(selectElement, paramName) {
			var select2Field = selectElement.find('select');
			
			if (select2Field.length) {
				select2Field.select2().on('select2:select', function (e) {
					var selectedElement = $(e.currentTarget),
						selectVal = selectedElement.val();
					
					selectElement.data(paramName, selectVal);
				});
			}
		}
		
		function resetSelect2Field(selectElement, selectParam, selectValue) {
			var select2Field = selectElement.find('select');
			
			select2Field.val(selectValue).trigger('change');
			selectElement.data(selectParam, selectValue);
		}
		
		function edgtfInitUndoQueryButton(queryHolder) {
			var undoQueryButton = queryHolder.find('.edgtf-undo-query-save'),
				resultHolder = queryHolder.find('.edgtf-filter-query-results');
			
			undoQueryButton.on('click', function () {
				resultHolder.html('<span class="fa fa-spinner fa-spin" aria-hidden="true"></span>');
				
				var ajaxData = {
					action: 'urbango_listing_set_listing_list_ajax_remove_query',
					query_id: undoQueryButton.data('query-id')
				};
				
				$.ajax({
					type: 'POST',
					data: ajaxData,
					url: edgtfGlobalVars.vars.edgtfAjaxUrl,
					success: function (data) {
						var response = JSON.parse(data);
						
						resultHolder.html('<span class="edgtf-filter-result-message">' + response.message + '</span>').append(response.data);
					}
				});
			});
		}
	}
	
	/**
	 * Initializes listing list relation terms switcher functionality
	 */
	function edgtfInitListingListRelationTermsSwitcher($reset) {
		var holder = $('.edgtf-ll-filter-holder');
		
		if (holder.length) {
			holder.each(function () {
				var thisHolder = $(this),
					categoryHolder = thisHolder.find('.edgtf-filter-section.edgtf-filter-section-category'),
					tagHolder = thisHolder.find('.edgtf-filter-section.edgtf-filter-section-tag');
				
				if (categoryHolder.length && tagHolder.length) {
					var categorySelect = categoryHolder.find('select'),
						tagItems = tagHolder.find('.edgtf-fs-cb-items'),
						tagItem = tagItems.children();
					
					if (tagItem.length > 24) { // 24 is sum of max items inside tags visible area
						tagItems.addClass('edgtf-all');
					}
					
					if ($reset) {
						tagItem.removeClass('edgtf-hide edgtf-show');
					}
					
					categorySelect.select2().on('select2:select', function (e) {
						var selectVal = $(e.currentTarget).val();
						
						tagItems.removeClass('edgtf-all').addClass('edgtf-loading').append('<span class="edgtf-loading-spinner fa fa-spinner fa-spin" aria-hidden="true"></span>');
						
						var ajaxData = {
							action: 'urbango_listing_init_listing_list_relation_terms_switcher',
							data: selectVal.length > 0 ? selectVal : '-1'
						};
						
						$.ajax({
							type: 'POST',
							data: ajaxData,
							url: edgtfGlobalVars.vars.edgtfAjaxUrl,
							success: function (data) {
								var response = JSON.parse(data),
									responseData = response.data;
								
								tagItem.each(function () {
									var thisTagItem = $(this),
										tagInputField = thisTagItem.find('input[type="checkbox"]');
									
									thisTagItem.addClass('edgtf-hide');
									
									$.each(responseData, function (index, value) {
										if (tagInputField.data('id') === parseInt(index, 10)) {
											thisTagItem.removeClass('edgtf-hide').addClass('edgtf-show');
										}
									});
								});
								
								if (selectVal.length <= 0 && tagItem.length > 24) {
									tagItems.addClass('edgtf-all');
								}
								
								tagItems.removeClass('edgtf-loading').find('.edgtf-loading-spinner').remove();
							}
						});
					});
				}
			});
		}
	}
	
	/**
	 * Initializes pagination functionality
	 */
	function edgtfInitListingListPagination() {
		var listingList = $('.edgtf-listing-list-holder');
		
		var initStandardPagination = function (thisListingList) {
			var standardLink = thisListingList.find('.edgtf-ll-standard-pagination li');
			
			if (standardLink.length) {
				standardLink.each(function () {
					var thisLink = $(this).children('a'),
						pagedLink = 1;
					
					thisLink.on('click', function (e) {
						e.preventDefault();
						e.stopPropagation();
						
						if (typeof thisLink.data('paged') !== 'undefined' && thisLink.data('paged') !== false) {
							pagedLink = thisLink.data('paged');
						}
						
						initMainPagFunctionality(thisListingList, pagedLink);
					});
				});
			}
		};
		
		var initLoadMorePagination = function (thisListingList) {
			var loadMoreButton = thisListingList.find('.edgtf-ll-load-more a');
			
			loadMoreButton.on('click', function (e) {
				e.preventDefault();
				e.stopPropagation();
				
				initMainPagFunctionality(thisListingList);
			});
		};
		
		var initInifiteScrollPagination = function (thisListingList) {
			var listingListHeight = thisListingList.outerHeight(),
				listingListTopOffest = thisListingList.offset().top,
				listingListPosition = listingListHeight + listingListTopOffest - edgtfGlobalVars.vars.edgtfAddForAdminBar;
			
			if (!thisListingList.hasClass('edgtf-ll-infinite-scroll-started') && edgtf.scroll + edgtf.windowHeight > listingListPosition) {
				initMainPagFunctionality(thisListingList);
			}
		};
		
		var initMainPagFunctionality = function (thisListingList, pagedLink, filterTrigger, locationObject) {
			var thisListingListInner = thisListingList.find('.edgtf-ll-inner'),
				nextPage,
				maxNumPages;
			
			if (typeof locationObject === 'object') {
				var currentLocation = thisListingList.data('location');
				
				if (currentLocation !== undefined && currentLocation.indexOf(locationObject[0]) > -1) {
					return;
				} else {
					thisListingList.data('location', locationObject[0]);
				}
			}
			
			if (typeof thisListingList.data('max-num-pages') !== 'undefined' && thisListingList.data('max-num-pages') !== false) {
				maxNumPages = thisListingList.data('max-num-pages');
			}
			
			if (thisListingList.hasClass('edgtf-ll-pag-standard')) {
				thisListingList.data('next-page', pagedLink);
			}
			
			if (thisListingList.hasClass('edgtf-ll-pag-infinite-scroll')) {
				thisListingList.addClass('edgtf-ll-infinite-scroll-started');
			}
			
			if (pagedLink === 1) {
				thisListingList.data('next-page', pagedLink);
			}
			
			var loadMoreData = edgtf.modules.common.getLoadMoreData(thisListingList),
				loadingItem = thisListingList.find('.edgtf-ll-loading');
			
			nextPage = loadMoreData.nextPage;
			if (nextPage <= maxNumPages || maxNumPages === 0) {
				if (filterTrigger && nextPage === 1) {
					loadingItem.css('top', thisListingList.find('.edgtf-ll-filter-holder').outerHeight() + 60).addClass('edgtf-showing edgtf-fade-trigger');
					thisListingList.addClass('edgtf-fade-trigger-animate');
				} else {
					if (thisListingList.hasClass('edgtf-ll-pag-standard')) {
						loadingItem.addClass('edgtf-showing edgtf-fade-trigger');
						thisListingList.addClass('edgtf-fade-trigger-animate');
					} else {
						loadingItem.addClass('edgtf-showing');
						thisListingList.find('.edgtf-ll-load-more-holder').hide();
					}
				}
				
				var ajaxData = edgtf.modules.common.setLoadMoreAjaxData(loadMoreData, 'urbango_listing_init_listing_list_ajax_load_more');
				
				$.ajax({
					type: 'POST',
					data: ajaxData,
					url: edgtfGlobalVars.vars.edgtfAjaxUrl,
					success: function (data) {
						if (!thisListingList.hasClass('edgtf-ll-pag-standard')) {
							nextPage++;
						}
						
						thisListingList.data('next-page', nextPage);
						
						var response = $.parseJSON(data),
							responseHtml = response.html;
						
						//get map items
						var mapObjs = response.mapAddresses,
							mapAddresses = '';
						
						if (mapObjs !== null && mapObjs['addresses'] !== undefined) {
							mapAddresses = mapObjs['addresses'];
						}
						
						if (thisListingList.hasClass('edgtf-ll-pag-standard') || pagedLink === 1) {
							edgtfInitStandardPaginationLinkChanges(thisListingList, maxNumPages, nextPage);
							
							thisListingList.waitForImages(function () {
								if (thisListingList.hasClass('edgtf-ll-masonry')) {
									edgtfInitHtmlIsotopeNewContent(thisListingList, thisListingListInner, loadingItem, responseHtml, mapAddresses, locationObject);
								} else if (thisListingList.hasClass('edgtf-ll-gallery') && thisListingList.hasClass('edgtf-ll-has-filter')) {
									edgtfInitHtmlIsotopeNewContent(thisListingList, thisListingListInner, loadingItem, responseHtml, mapAddresses, locationObject);
								} else {
									edgtfInitHtmlGalleryNewContent(thisListingList, thisListingListInner, loadingItem, responseHtml, mapAddresses, locationObject);
								}
							});
						} else {
							thisListingList.waitForImages(function () {
								if (thisListingList.hasClass('edgtf-ll-masonry')) {
									if (pagedLink === 1) {
										edgtfInitHtmlIsotopeNewContent(thisListingList, thisListingListInner, loadingItem, responseHtml, mapAddresses, locationObject);
									} else {
										edgtfInitAppendIsotopeNewContent(thisListingList, thisListingListInner, loadingItem, responseHtml, mapAddresses, locationObject);
									}
								} else if (thisListingList.hasClass('edgtf-ll-gallery') && thisListingList.hasClass('edgtf-ll-has-filter') && pagedLink !== 1) {
									edgtfInitAppendIsotopeNewContent(thisListingList, thisListingListInner, loadingItem, responseHtml, mapAddresses, locationObject);
								} else {
									edgtfInitAppendGalleryNewContent(thisListingList, thisListingListInner, loadingItem, responseHtml, mapAddresses, locationObject);
								}
							});
						}
						
						if (thisListingList.hasClass('edgtf-ll-infinite-scroll-started')) {
							thisListingList.removeClass('edgtf-ll-infinite-scroll-started');
						}
						
						if (response.postsCount < 1 || response.postsCount < thisListingList.data('number-of-items')) {
							thisListingList.find('.edgtf-ll-load-more-holder').hide();
						} else {
							thisListingList.find('.edgtf-ll-load-more-holder').show();
						}
					}
				});
			}
			
			if (pagedLink === 1) {
				thisListingList.find('.edgtf-ll-load-more-holder').show();
			}
			
			if (nextPage === maxNumPages) {
				thisListingList.find('.edgtf-ll-load-more-holder').hide();
			}
		};
		
		var edgtfInitStandardPaginationLinkChanges = function (thisListingList, maxNumPages, nextPage) {
			var standardPagHolder = thisListingList.find('.edgtf-ll-standard-pagination'),
				standardPagNumericItem = standardPagHolder.find('li.edgtf-ll-pag-number'),
				standardPagPrevItem = standardPagHolder.find('li.edgtf-ll-pag-prev a'),
				standardPagNextItem = standardPagHolder.find('li.edgtf-ll-pag-next a');
			
			standardPagNumericItem.removeClass('edgtf-ll-pag-active');
			standardPagNumericItem.eq(nextPage - 1).addClass('edgtf-ll-pag-active');
			
			standardPagPrevItem.data('paged', nextPage - 1);
			standardPagNextItem.data('paged', nextPage + 1);
			
			if (nextPage > 1) {
				standardPagPrevItem.css({'opacity': '1'});
			} else {
				standardPagPrevItem.css({'opacity': '0'});
			}
			
			if (nextPage === maxNumPages) {
				standardPagNextItem.css({'opacity': '0'});
			} else {
				standardPagNextItem.css({'opacity': '1'});
			}
		};
		
		var edgtfInitHtmlIsotopeNewContent = function (thisListingList, thisListingListInner, loadingItem, responseHtml, mapAddresses, locationObject) {
			thisListingListInner.find('article').remove();
			thisListingListInner.append(responseHtml);
			edgtf.modules.common.setFixedImageProportionSize(thisListingList, thisListingList.find('article'), thisListingListInner.find('.edgtf-ll-grid-sizer').width());
			
			thisListingListInner.isotope('reloadItems').isotope({sortBy: 'original-order'});
			
			edgtfReInitResultsPostCount(thisListingList, thisListingListInner);
			edgtfReInitMapFunctionality(thisListingList, mapAddresses, locationObject, 'append');
			
			loadingItem.removeClass('edgtf-showing edgtf-fade-trigger');
			thisListingList.removeClass('edgtf-fade-trigger-animate');
			
			setTimeout(function () {
				thisListingListInner.isotope('layout');
				edgtf.modules.common.edgtfInitParallax();
				edgtfListHoverClass();
			}, 600);
		};
		
		var edgtfInitHtmlGalleryNewContent = function (thisListingList, thisListingListInner, loadingItem, responseHtml, mapAddresses, locationObject) {
			loadingItem.removeClass('edgtf-showing edgtf-fade-trigger');
			thisListingList.removeClass('edgtf-fade-trigger-animate');
			thisListingListInner.html(responseHtml);
			
			edgtfReInitResultsPostCount(thisListingList, thisListingListInner);
			edgtfReInitMapFunctionality(thisListingList, mapAddresses, locationObject, 'replace');
			
			edgtf.modules.common.edgtfInitParallax();
			edgtfListHoverClass();
		};
		
		var edgtfInitAppendIsotopeNewContent = function (thisListingList, thisListingListInner, loadingItem, responseHtml, mapAddresses, locationObject) {
			thisListingListInner.append(responseHtml);
			edgtf.modules.common.setFixedImageProportionSize(thisListingList, thisListingList.find('article'), thisListingListInner.find('.edgtf-ll-grid-sizer').width());
			
			thisListingListInner.isotope('reloadItems').isotope({sortBy: 'original-order'});
			
			edgtfReInitResultsPostCount(thisListingList, thisListingListInner);
			edgtfReInitMapFunctionality(thisListingList, mapAddresses, locationObject, 'append');
			
			loadingItem.removeClass('edgtf-showing');
			
			setTimeout(function () {
				thisListingListInner.isotope('layout');
				edgtf.modules.common.edgtfInitParallax();
				edgtfListHoverClass();
			}, 600);
		};
		
		var edgtfInitAppendGalleryNewContent = function (thisListingList, thisListingListInner, loadingItem, responseHtml, mapAddresses, locationObject) {
			loadingItem.removeClass('edgtf-showing');
			thisListingListInner.append(responseHtml);
			
			edgtfReInitResultsPostCount(thisListingList, thisListingListInner);
			edgtfReInitMapFunctionality(thisListingList, mapAddresses, locationObject, 'append');
			
			edgtf.modules.common.edgtfInitParallax();
			edgtfListHoverClass();
		};
		
		var edgtfReInitResultsPostCount = function (thisListingList, thisListingListInner) {
			if (thisListingList.hasClass('edgtf-ll-with-filter')) {
				var resultsPostCount = thisListingList.find('.edgtf-filter-section-switch-layout .edgtf-fs-sl-count');
				
				if (resultsPostCount.length) {
					var postsCount = parseInt(thisListingListInner.find('article').length, 10);
					
					resultsPostCount.html(postsCount);
				}
			}
		};
		
		var edgtfReInitMapFunctionality = function (thisListingList, mapAddresses, locationObject, type) {
			if (thisListingList.hasClass('edgtf-ll-with-map')) {
				if (mapAddresses !== '') {
					edgtf.modules.listingMaps.edgtfReinitMultipleGoogleMaps(mapAddresses, type);
				}
				
				if (typeof locationObject === 'object') {
					edgtf.modules.listingMaps.edgtfGoogleMaps.centerOnForwardLocation(locationObject[1], locationObject[2], locationObject[0]);
				}
			}
		};
		
		return {
			init: function () {
				if (listingList.length) {
					listingList.each(function () {
						var thisPortList = $(this);
						
						if (thisPortList.hasClass('edgtf-ll-pag-standard')) {
							initStandardPagination(thisPortList);
						}
						
						if (thisPortList.hasClass('edgtf-ll-pag-load-more')) {
							initLoadMorePagination(thisPortList);
						}
						
						if (thisPortList.hasClass('edgtf-ll-pag-infinite-scroll')) {
							initInifiteScrollPagination(thisPortList);
						}
					});
				}
			},
			scroll: function () {
				if (listingList.length) {
					listingList.each(function () {
						var thisListingList = $(this);
						
						if (thisListingList.hasClass('edgtf-ll-pag-infinite-scroll')) {
							initInifiteScrollPagination(thisListingList);
						}
					});
				}
			},
			getMainPagFunction: function (thisListingList, paged, filterTrigger, geoLocationObject) {
				initMainPagFunctionality(thisListingList, paged, filterTrigger, geoLocationObject);
			}
		};
	}

	function edgtfListHoverClass() {
		var lists = $('.edgtf-listing-list-holder');

		lists.length && edgtf.modules.common.edgtfToggleHoverClasses('.edgtf-listing-list-holder', 'article', '.edgtf-lli-image');
    }
	
})(jQuery);
(function ($) {
	'use strict';
	
	var listingSearch = {};
	edgtf.modules.listingSearch = listingSearch;
	
	listingSearch.edgtfOnDocumentReady = edgtfOnDocumentReady;
	
	$(document).ready(edgtfOnDocumentReady);
	
	/*
	 All functions to be called on $(document).ready() should be in this function
	 */
	function edgtfOnDocumentReady() {
		initListingSearchParams();
	}
	
	function initListingSearchParams() {
		var holder = $('.edgtf-listing-search-holder');
		
		if (holder.length) {
			holder.each(function () {
				var thisHolder = $(this),
					selectInput = thisHolder.find('.edgtf-ls-form-section select'),
					locationInput = thisHolder.find('.edgtf-ls-form-location input[type="text"]'),
					locationInputID = document.getElementById('edgtf-ls-form-location-id'),
					geoLocationResetLink = thisHolder.find('.edgtf-ls-places-reset'),
					geoLocationLink = thisHolder.find('.edgtf-ls-places-geo'),
					geoLocationLinkIcon = geoLocationLink.children('.edgtf-ls-places-icon');
				
				if (selectInput.length) {
					selectInput.each(function () {
						var thisSelectInput = $(this);
						
						if (thisSelectInput.hasClass('edgtf-select-options-with-icons')) {
							thisSelectInput.select2({
								templateResult: setNewSelectOptionValue
							});
						} else {
							thisSelectInput.select2();
						}
					});
				}
				
				if (locationInput.length) {
					edgtf.modules.listingMaps.edgtfGoogleMaps.createAutocompletePlaces(locationInputID);
				}
				
				geoLocationResetLink.on('click', function (e) {
					e.preventDefault();
					
					locationInput.val('');
					geoLocationResetLink.hide();
				});
				
				geoLocationLink.on('click', function (e) {
					e.preventDefault();
					
					if (typeof google === 'object' && typeof navigator === 'object') {
						geoLocationResetLink.show();
						edgtf.modules.listingMaps.edgtfGoogleMaps.centerOnCurrentLocation(true, locationInput, geoLocationLinkIcon);
					}
				});
			});
		}
	}
	
	function setNewSelectOptionValue(state) {
		if (!state.id) {
			return state.text;
		}
		
		var originalElement = state.element;
		
		if ($(originalElement).data('custom-image') === 'undefined' || $(originalElement).data('custom-image') === false || $(originalElement).data('custom-image') === '') {
			return state.text;
		}
		
		var $new_text = state.text === 'all' ? '' : state.text,
			$new_state = $('<span class="edgtf-select2-with-image"><img src="' + $(originalElement).data('custom-image') + '" alt="' + $(originalElement).data('custom-image-alt') + '"/>' + $new_text + '</span>');
		
		return $new_state;
	}
	
})(jQuery);
(function ($) {
	'use strict';
	
	var edgtfOT = {};
	edgtf.modules.edgtfOT = edgtfOT;
	
	edgtfOT.edgtfOnDocumentReady = edgtfOnDocumentReady;
	
	$(document).ready(edgtfOnDocumentReady);
	
	/*
	 All functions to be called on $(document).ready() should be in this function
	 */
	function edgtfOnDocumentReady() {
		initedgtfOTParams();
		reInitListingGalleryPrettyPhoto();
	}
	
	function initedgtfOTParams() {
		var holder = $('.edgtf-ls-open-table');
		
		if (holder.length) {
			holder.each(function () {
				var thisHolder = $(this),
					selectInputFields = thisHolder.find('select');
				
				if (selectInputFields.length) {
					selectInputFields.each(function () {
						$(this).select2();
					});
				}
			});
		}
	}

	function reInitListingGalleryPrettyPhoto() {
		var gallerySlider = $( '.edgtf-ls-gallery-images.edgtf-owl-slider' );

		if ( gallerySlider.length ) {
			gallerySlider.on(
				'changed.owl.carousel',
				function () {
					edgtf.modules.common.edgtfPrettyPhoto();
				}
			);
		}
	}
	
})(jQuery);