<?php

namespace UrbanGoListing\CPT\Shortcodes\Listing;

use UrbanGoListing\Lib;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

class LocationList implements Lib\ShortcodeInterface {
	private $base;
	
	public function __construct() {
		$this->base = 'edgtf_location_list';
		
		add_action( 'vc_before_init', array( $this, 'vcMap' ) );
	}
	
	public function getBase() {
		return $this->base;
	}
	
	public function vcMap() {
		if ( function_exists( 'vc_map' ) ) {
			vc_map( array(
					'name'     => esc_html__( 'Location List', 'urbango-listing' ),
					'base'     => $this->getBase(),
					'category' => esc_html__( 'by URBANGO LISTING', 'urbango-listing' ),
					'icon'     => 'icon-wpb-location-list extended-custom-listing-icon',
					'params'   => array(
						array(
							'type'        => 'dropdown',
							'param_name'  => 'number_of_columns',
							'heading'     => esc_html__( 'Number of Columns', 'urbango-listing' ),
							'description' => esc_html__( 'Set number of columns for your list. Default value is three', 'urbango-listing' ),
							'value'       => array_flip( urbango_edge_get_number_of_columns_array( true ) )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'space_between_items',
							'heading'     => esc_html__( 'Space Between Items', 'urbango-listing' ),
							'description' => esc_html__( 'Set space size between items for your list. Default value is normal', 'urbango-listing' ),
							'value'       => array_flip( urbango_edge_get_space_between_items_array() )
						),
						array(
							'type'        => 'textfield',
							'param_name'  => 'number_of_items',
							'heading'     => esc_html__( 'Number of Items', 'urbango-listing' ),
							'description' => esc_html__( 'Set number of items for your list', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'location',
							'heading'     => esc_html__( 'Location', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_cpt_taxonomy_items( 'location' ) ),
							'save_always' => true
						),
						array(
							'type'        => 'textfield',
							'param_name'  => 'location_slugs',
							'heading'     => esc_html__( 'Location Slugs', 'urbango-listing' ),
							'description' => esc_html__( 'Enter the location slugs you want to display, separated by commas (leave empty for showing all locations)', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'order_by',
							'heading'     => esc_html__( 'Order By', 'urbango-listing' ),
							'value'       => array_flip( urbango_edge_get_query_order_by_array() ),
                            'save_always' => true
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'order',
							'heading'     => esc_html__( 'Order', 'urbango-listing' ),
							'value'       => array_flip( urbango_edge_get_query_order_array() ),
                            'save_always' => true
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'image_proportions',
							'heading'     => esc_html__( 'Image Proportions', 'urbango-listing' ),
							'value'       => array(
								esc_html__( 'Original', 'urbango-listing' )  => 'full',
								esc_html__( 'Square', 'urbango-listing' )    => 'square',
								esc_html__( 'Landscape', 'urbango-listing' ) => 'landscape',
								esc_html__( 'Portrait', 'urbango-listing' )  => 'portrait',
								esc_html__( 'Thumbnail', 'urbango-listing' ) => 'thumbnail',
								esc_html__( 'Medium', 'urbango-listing' )    => 'medium',
								esc_html__( 'Large', 'urbango-listing' )     => 'large'
							),
							'description' => esc_html__( 'Set image proportions for your list. Default value is full', 'urbango-listing' ),
							'group'       => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'title_tag',
							'heading'    => esc_html__( 'Title Tag', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_title_tag( true, array( 'p' => 'p' ) ) ),
							'dependency' => array( 'element' => 'enable_title', 'value' => array( 'yes' ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_excerpt',
							'heading'    => esc_html__( 'Enable Excerpt', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'        => 'textfield',
							'param_name'  => 'excerpt_length',
							'heading'     => esc_html__( 'Excerpt Length', 'urbango-listing' ),
							'description' => esc_html__( 'Set number of characters', 'urbango-listing' ),
							'dependency'  => array( 'element' => 'enable_excerpt', 'value' => array( 'yes' ) ),
							'group'       => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_items_count',
							'heading'    => esc_html__( 'Enable Items Count', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						)
					)
				)
			);
		}
	}
	
	public function render( $atts, $content = null ) {
		$args   = array(
			'number_of_columns'   => 'three',
			'space_between_items' => 'normal',
			'number_of_items'     => '9',
			'location'            => '',
			'location_slugs'      => '',
			'order_by'            => 'date',
			'order'               => 'DESC',
			'image_proportions'   => 'full',
			'title_tag'           => 'h5',
			'enable_excerpt'      => 'yes',
			'excerpt_length'      => '',
			'enable_items_count'  => 'yes'
		);
		$params = shortcode_atts( $args, $atts );
		
		/***
		 * @params query_results
		 * @params holder_classes
		 */
		$additional_params = array();
		
		$additional_params['query_results']  = $this->getQueryArray( $params );
		$additional_params['holder_classes'] = $this->getHolderClasses( $params, $args );
		
		$html = urbango_listing_get_module_template_part( 'cpt-listing', 'location-list', 'holder', '', $params, $additional_params, false );
		
		return $html;
	}
	
	/**
	 * Generates query results
	 *
	 * @param $params
	 *
	 * @return array
	 */
	public function getQueryArray( $params ) {
		$query_args = array(
			'taxonomy' => 'listing-location',
			'number'   => $params['number_of_items'],
			'orderby'  => $params['order_by'],
			'order'    => $params['order']
		);
		
		$location = trim( $params['location'] );
		if ( ! empty( $location ) ) {
			$query_args['include'] = array( intval( $location ) );
		}
		
		$location_slugs = trim( $params['location_slugs'] );
		if ( ! empty( $location_slugs ) && empty( $location ) ) {
			$query_args['slug'] = explode( ',', str_replace( ' ', '', $location_slugs ) );
		}
		
		$query_array = get_terms( $query_args );


		return $query_array;
	}
	
	/**
	 * Generates holder classes
	 *
	 * @param $params
	 * @param $args
	 *
	 * @return string
	 */
	public function getHolderClasses( $params, $args ) {
		$classes = array();
		
		$classes[] = ! empty( $params['number_of_columns'] ) ? 'edgtf-' . $params['number_of_columns'] . '-columns' : 'edgtf-' . $args['number_of_columns'] . '-columns';
		$classes[] = ! empty( $params['space_between_items'] ) ? 'edgtf-' . $params['space_between_items'] . '-space' : 'edgtf-' . $args['space_between_items'] . '-space';
		
		return implode( ' ', $classes );
	}
	
	/**
	 * Generates image size
	 *
	 * @param $params
	 *
	 * @return string
	 */
	public function getImageSize( $params ) {
		$thumb_size = 'full';
		
		if ( ! empty( $params['image_proportions'] ) ) {
			$image_size = $params['image_proportions'];
			
			switch ( $image_size ) {
				case 'landscape':
					$thumb_size = 'urbango_edge_image_landscape';
					break;
				case 'portrait':
					$thumb_size = 'urbango_edge_image_portrait';
					break;
				case 'square':
					$thumb_size = 'urbango_edge_image_square';
					break;
				case 'thumbnail':
					$thumb_size = 'thumbnail';
					break;
				case 'medium':
					$thumb_size = 'medium';
					break;
				case 'large':
					$thumb_size = 'large';
					break;
				case 'full':
					$thumb_size = 'full';
					break;
			}
		}
		
		return $thumb_size;
	}
}
