<?php

namespace UrbanGoListing\CPT\Shortcodes\Listing;

use UrbanGoListing\Lib;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

class ListingList implements Lib\ShortcodeInterface {
	private $base;
	
	public function __construct() {
		$this->base = 'edgtf_listing_list';
		
		// Listing selected items filter
		add_filter( 'vc_autocomplete_edgtf_listing_list_selected_items_callback', array( &$this, 'listingIdAutocompleteSuggester', ), 10, 1 ); // Get suggestion(find). Must return an array
		
		// Listing selected items render
		add_filter( 'vc_autocomplete_edgtf_listing_list_selected_items_render', array( &$this, 'listingIdAutocompleteRender', ), 10, 1 ); // Render exact listing. Must return an array (label,value)
		
		add_action( 'vc_before_init', array( $this, 'vcMap' ) );
	}
	
	public function getBase() {
		return $this->base;
	}
	
	public function vcMap() {
		if ( function_exists( 'vc_map' ) ) {
			vc_map( array(
					'name'     => esc_html__( 'Listing List', 'urbango-listing' ),
					'base'     => $this->getBase(),
					'category' => esc_html__( 'by URBANGO LISTING', 'urbango-listing' ),
					'icon'     => 'icon-wpb-listing-list extended-custom-listing-icon',
					'params'   => array(
						array(
							'type'        => 'textfield',
							'param_name'  => 'custom_class',
							'heading'     => esc_html__( 'Custom CSS Class', 'urbango-listing' ),
							'description' => esc_html__( 'Style particular content element differently - add a class name and refer to it in custom CSS', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'type',
							'heading'     => esc_html__( 'Type', 'urbango-listing' ),
							'value'       => array(
								esc_html__( 'Gallery', 'urbango-listing' ) => 'gallery',
								esc_html__( 'Masonry', 'urbango-listing' ) => 'masonry'
							),
							'save_always' => true,
							'admin_label' => true
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'item_layout',
							'heading'     => esc_html__( 'Item Layout', 'urbango-listing' ),
							'value'       => array(
								esc_html__( 'Standard', 'urbango-listing' ) => 'standard',
								esc_html__( 'Simple', 'urbango-listing' )   => 'simple'
							),
							'save_always' => true
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'number_of_columns',
							'heading'     => esc_html__( 'Number of Columns', 'urbango-listing' ),
							'description' => esc_html__( 'Set number of columns for your list. Default value is three', 'urbango-listing' ),
							'value'       => array_flip( urbango_edge_get_number_of_columns_array( true ) )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'space_between_items',
							'heading'     => esc_html__( 'Space Between Items', 'urbango-listing' ),
							'description' => esc_html__( 'Set space size between items for your list. Default value is normal', 'urbango-listing' ),
							'value'       => array_flip( urbango_edge_get_space_between_items_array() )
						),
						array(
							'type'        => 'textfield',
							'param_name'  => 'number_of_items',
							'heading'     => esc_html__( 'Number of Items', 'urbango-listing' ),
							'description' => esc_html__( 'Set number of items for your list. Enter -1 to show all.', 'urbango-listing' ),
							'value'       => '-1'
						),
						array(
							'type'        => 'autocomplete',
							'param_name'  => 'selected_items',
							'heading'     => esc_html__( 'Show Only Items with Listed IDs', 'urbango-listing' ),
							'settings'    => array(
								'multiple'      => true,
								'sortable'      => true,
								'unique_values' => true
							),
							'description' => esc_html__( 'Delimit ID numbers by comma (leave empty for all)', 'urbango-listing' ),
							'group'       => esc_html__( 'Query Options', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'order_by',
							'heading'     => esc_html__( 'Order By', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_listing_list_query_order_by_array() ),
							'group'       => esc_html__( 'Query Options', 'urbango-listing' ),
                            'save_always' => true,
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'order',
							'heading'     => esc_html__( 'Order', 'urbango-listing' ),
							'value'       => array_flip( urbango_edge_get_query_order_array() ),
							'group'       => esc_html__( 'Query Options', 'urbango-listing' ),
                            'save_always' => true,
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'category',
							'heading'     => esc_html__( 'By Category', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_cpt_taxonomy_items( 'category' ) ),
							'save_always' => true,
							'group'       => esc_html__( 'Query Options', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'location',
							'heading'     => esc_html__( 'By Location', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_cpt_taxonomy_items( 'location' ) ),
							'save_always' => true,
							'group'       => esc_html__( 'Query Options', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'tag',
							'heading'     => esc_html__( 'By Tag', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_cpt_taxonomy_items( 'tag' ) ),
							'save_always' => true,
							'group'       => esc_html__( 'Query Options', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'amenity',
							'heading'     => esc_html__( 'By Amenity', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_cpt_taxonomy_items( 'amenity' ) ),
							'save_always' => true,
							'group'       => esc_html__( 'Query Options', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'image_proportions',
							'heading'     => esc_html__( 'Image Proportions', 'urbango-listing' ),
							'value'       => array(
								esc_html__( 'Original', 'urbango-listing' )  => 'full',
								esc_html__( 'Square', 'urbango-listing' )    => 'square',
								esc_html__( 'Landscape', 'urbango-listing' ) => 'landscape',
								esc_html__( 'Portrait', 'urbango-listing' )  => 'portrait',
								esc_html__( 'Thumbnail', 'urbango-listing' ) => 'thumbnail',
								esc_html__( 'Medium', 'urbango-listing' )    => 'medium',
								esc_html__( 'Large', 'urbango-listing' )     => 'large'
							),
							'description' => esc_html__( 'Set image proportions for your list. Default value is full', 'urbango-listing' ),
							'dependency'  => array( 'element' => 'type', 'value' => array( 'gallery' ) ),
							'group'       => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'title_tag',
							'heading'    => esc_html__( 'Title Tag', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_title_tag( true, array( 'p' => 'p' ) ) ),
							'dependency' => array( 'element' => 'enable_title', 'value' => array( 'yes' ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_excerpt',
							'heading'    => esc_html__( 'Enable Excerpt', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'        => 'textfield',
							'param_name'  => 'excerpt_length',
							'heading'     => esc_html__( 'Excerpt Length', 'urbango-listing' ),
							'description' => esc_html__( 'Set number of characters', 'urbango-listing' ),
							'dependency'  => array( 'element' => 'enable_excerpt', 'value' => array( 'yes' ) ),
							'group'       => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_category',
							'heading'    => esc_html__( 'Enable Category', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_location',
							'heading'    => esc_html__( 'Enable Location', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_reviews_count',
							'heading'    => esc_html__( 'Overall Rating', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_price_range',
							'heading'    => esc_html__( 'Enable Price Range', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_map',
							'heading'    => esc_html__( 'Enable Map', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false ) ),
							'group'      => esc_html__( 'Additional Features', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_map_switcher',
							'heading'    => esc_html__( 'Enable Map Switcher', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false ) ),
							'dependency' => array( 'element' => 'enable_map', 'value' => array( 'yes' ) ),
							'group'      => esc_html__( 'Additional Features', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_filter',
							'heading'    => esc_html__( 'Enable Filter', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false ) ),
							'group'      => esc_html__( 'Additional Features', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_filter_custom_search',
							'heading'    => esc_html__( 'Enable Filter - Custom Search', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'dependency' => array( 'element' => 'enable_filter', 'value' => array( 'yes' ) ),
							'group'      => esc_html__( 'Additional Features', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_filter_category',
							'heading'    => esc_html__( 'Enable Filter - Category', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'dependency' => array( 'element' => 'enable_filter', 'value' => array( 'yes' ) ),
							'group'      => esc_html__( 'Additional Features', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_filter_location',
							'heading'    => esc_html__( 'Enable Filter - Location', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'dependency' => array( 'element' => 'enable_filter', 'value' => array( 'yes' ) ),
							'group'      => esc_html__( 'Additional Features', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'filter_location_field_type',
							'heading'     => esc_html__( 'Filter Location Field Type', 'urbango-listing' ),
							'description' => esc_html__( 'Choose filter location input field type', 'urbango-listing' ),
							'value'       => array(
								esc_html__( 'Select Box', 'urbango-listing' )             => 'select',
								esc_html__( 'AutoComplete Locations', 'urbango-listing' ) => 'places'
							),
							'dependency'  => array( 'element' => 'enable_filter_location', 'value' => array( 'yes' ) ),
							'group'       => esc_html__( 'Additional Features', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_filter_tag',
							'heading'    => esc_html__( 'Enable Filter - Tag', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'dependency' => array( 'element' => 'enable_filter', 'value' => array( 'yes' ) ),
							'group'      => esc_html__( 'Additional Features', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_filter_order_by',
							'heading'    => esc_html__( 'Enable Filter - Order By', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'dependency' => array( 'element' => 'enable_filter', 'value' => array( 'yes' ) ),
							'group'      => esc_html__( 'Additional Features', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'enable_filter_switch_layout',
							'heading'    => esc_html__( 'Enable Filter - Switch Layout', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_yes_no_select_array( false, true ) ),
							'dependency' => array( 'element' => 'enable_filter', 'value' => array( 'yes' ) ),
							'group'      => esc_html__( 'Additional Features', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'pagination_type',
							'heading'    => esc_html__( 'Pagination Type', 'urbango-listing' ),
							'value'      => array(
								esc_html__( 'None', 'urbango-listing' )            => 'no-pagination',
								esc_html__( 'Standard', 'urbango-listing' )        => 'standard',
								esc_html__( 'Load More', 'urbango-listing' )       => 'load-more',
								esc_html__( 'Infinite Scroll', 'urbango-listing' ) => 'infinite-scroll'
							),
							'group'      => esc_html__( 'Additional Features', 'urbango-listing' )
						),
						array(
							'type'       => 'textfield',
							'param_name' => 'load_more_top_margin',
							'heading'    => esc_html__( 'Load More Top Margin (px or %)', 'urbango-listing' ),
							'dependency' => array( 'element' => 'pagination_type', 'value' => array( 'load-more' ) ),
							'group'      => esc_html__( 'Additional Features', 'urbango-listing' )
						)
					)
				)
			);
		}
	}
	
	public function render( $atts, $content = null ) {
		$args   = array(
			'custom_class'                => '',
			'type'                        => 'gallery',
			'item_layout'                 => 'standard',
			'number_of_columns'           => 'three',
			'space_between_items'         => 'normal',
			'number_of_items'             => '-1',
			'selected_items'              => '',
			'order_by'                    => 'date',
			'order'                       => 'ASC',
			'custom_search'               => '',
			'category'                    => '',
			'location'                    => '',
			'tag'                         => '',
			'amenity'                     => '',
			'author_id'                   => '',
			'image_proportions'           => 'full',
			'title_tag'                   => 'h5',
			'enable_excerpt'              => 'yes',
			'excerpt_length'              => '',
			'enable_category'             => 'yes',
			'enable_location'             => 'yes',
			'enable_reviews_count'        => 'yes',
			'enable_price_range'          => 'yes',
			'enable_map'                  => 'no',
			'enable_map_switcher'         => 'no',
			'enable_filter'               => 'no',
			'enable_filter_custom_search' => 'yes',
			'enable_filter_category'      => 'yes',
			'enable_filter_location'      => 'yes',
			'filter_location_field_type'  => 'select',
			'enable_filter_tag'           => 'yes',
			'enable_filter_order_by'      => 'yes',
			'enable_filter_switch_layout' => 'yes',
			'pagination_type'             => 'no-pagination',
			'load_more_top_margin'        => '',
			'hide_active_filter'          => 'yes'
		);
		$params = shortcode_atts( $args, $atts );
		
		/***
		 * @params query_results
		 * @params holder_data
		 * @params holder_classes
		 * @params holder_inner_classes
		 */
		$additional_params = array();
		
		$query_array   = $this->getQueryArray( $params );
		$query_results = new \WP_Query( $query_array );
		$posts_count   = $query_results->post_count;
		
		$additional_params['query_array']          = $query_array;
		$additional_params['query_results']        = $query_results;
		$additional_params['holder_data']          = urbango_edge_get_holder_data_for_cpt( $params, $additional_params );
		$additional_params['holder_classes']       = $this->getHolderClasses( $params, $args );
		$additional_params['holder_inner_classes'] = $this->getHolderInnerClasses( $params );
		
		$params['posts_count'] = $posts_count;
		$params['this_object'] = $this;
		
		$html = urbango_listing_get_module_template_part( 'cpt-listing', 'listing-list', 'holder', $params['type'], $params, $additional_params, false );
		
		return $html;
	}
	
	/**
	 * Generates query results
	 *
	 * @param $params
	 *
	 * @return array
	 */
	public function getQueryArray( $params ) {
		$query_array = array(
			'post_status'    => 'publish',
			'post_type'      => 'listing-item',
			'posts_per_page' => $params['number_of_items'],
			'order'          => $params['order']
		);
		
		$items_ids = null;
		if ( ! empty( $params['selected_items'] ) ) {
			$items_ids               = explode( ',', $params['selected_items'] );
			$query_array['post__in'] = $items_ids;
		}
		
		// Taxonomy query values
		if ( ! empty( $params['category'] ) || ! empty( $params['location'] ) || ! empty( $params['tag'] ) || ! empty( $params['amenity'] ) ) {
			$tax_query = array();
			
			if ( ! empty( $params['category'] ) ) {
				$tax_query[] = array(
					'taxonomy' => 'listing-category',
					'terms'    => $params['category']
				);
			}
			
			if ( ! empty( $params['location'] ) ) {
				$location = esc_attr( $params['location'] );
				
				if ( is_int( intval( $location ) ) && intval( $location ) > 0 ) {
					$tax_query[] = array(
						'taxonomy' => 'listing-location',
						'terms'    => $location
					);
				} else {
					$tax_query[] = array(
						'taxonomy' => 'listing-location',
						'field'    => 'slug',
						'terms'    => strtolower( $location ),
						'compare'  => 'LIKE'
					);
				}
			}
			
			if ( ! empty( $params['tag'] ) ) {
				$tax_query[] = array(
					'taxonomy' => 'listing-tag',
					'terms'    => $params['tag']
				);
			}
			
			if ( ! empty( $params['amenity'] ) ) {
				$tax_query[] = array(
					'taxonomy' => 'listing-amenity',
					'terms'    => $params['amenity']
				);
			}
			
			$query_array['tax_query'] = $tax_query;
		}
		
		// Custom meta query values
		if ( ! empty( $params['custom_search'] ) ) {
			$query_array['s'] = esc_attr( $params['custom_search'] );
		}
		
		// Custom meta query args
		$order_by   = ! empty( $params['order_by'] ) ? $params['order_by'] : 'date';
		$query_args = array( 'relation' => 'AND' );
		
		// Custom order meta query args
		if ( $order_by === 'featured-first' || $order_by === 'featured-last' ) {
			$query_array['orderby'] = 'edgtf_listing_is_featured_meta post_date';
			$query_array['order']   = $order_by === 'featured-first' ? 'DESC' : 'ASC';
			
			$query_args[] = array(
				'relation' => 'OR',
				array(
					'key'     => 'edgtf_listing_is_featured_meta',
					'value'   => array( 'yes', 'no', '' ),
					'compare' => 'IN'
				),
				array(
					'key'     => 'edgtf_listing_is_featured_meta',
					'compare' => 'NOT EXISTS'
				)
			);
		} else if ( $order_by === 'price-range-high' ) {
			$query_array['orderby'] = 'meta_value';
			
			$query_args[] = array(
				'key'     => 'edgtf_listing_single_price_range_meta',
				'value'   => '5',
				'compare' => '<='
			);
		} else if ( $order_by === 'price-range-low' ) {
			$query_array['orderby'] = 'meta_value';
			$query_array['order']   = 'ASC';
			
			$query_args[] = array(
				'key'     => 'edgtf_listing_single_price_range_meta',
				'value'   => '1',
				'compare' => '>='
			);
		} else {
			$query_array['orderby'] = $order_by;
		}
		
		// Author ID meta query args
		if ( ! empty( $params['author_id'] ) ) {
			$query_array['author'] = $params['author_id'];
		}
		
		$query_array['meta_query'] = $query_args;
		
		if ( ! empty( $params['next_page'] ) ) {
			$query_array['paged'] = $params['next_page'];
		} else {
			$query_array['paged'] = 1;
		}
		
		return $query_array;
	}
	
	/**
	 * Generates holder classes
	 *
	 * @param $params
	 * @param $args
	 *
	 * @return string
	 */
	public function getHolderClasses( $params, $args ) {
		$classes = array();
		
		$classes[] = ! empty( $params['custom_class'] ) ? esc_attr( $params['custom_class'] ) : '';
		$classes[] = ! empty( $params['type'] ) ? 'edgtf-ll-' . $params['type'] : 'edgtf-ll-' . $args['type'];
		$classes[] = ! empty( $params['item_layout'] ) ? 'edgtf-ll-layout-' . $params['item_layout'] : 'edgtf-ll-layout-' . $args['item_layout'];
		
		$classes[] = 'edgtf-grid-list edgtf-disable-bottom-space';
		$classes[] = $params['type'] === 'masonry' ? 'edgtf-grid-masonry-list' : '';
		
		$classes[] = ! empty( $params['number_of_columns'] ) ? 'edgtf-' . $params['number_of_columns'] . '-columns' : 'edgtf-' . $args['number_of_columns'] . '-columns';
		$classes[] = ! empty( $params['space_between_items'] ) ? 'edgtf-' . $params['space_between_items'] . '-space' : 'edgtf-' . $args['space_between_items'] . '-space';
		
		$classes[] = $params['enable_map'] === 'yes' ? 'edgtf-ll-with-map edgtf-map-list-holder' : 'edgtf-ll-no-map';
		$classes[] = $params['enable_map_switcher'] === 'yes' ? 'edgtf-ll-have-switcher' : '';
		$classes[] = $params['enable_filter'] === 'yes' ? 'edgtf-ll-with-filter' : 'edgtf-ll-no-filter';
		$classes[] = ! empty( $params['pagination_type'] ) ? 'edgtf-ll-pag-' . $params['pagination_type'] : '';
		
		$classes[] = ! empty( $params['category'] ) ? 'edgtf-ll-category-set' : '';
		$classes[] = ! empty( $params['location'] ) ? 'edgtf-ll-location-set' : '';
		$classes[] = ! empty( $params['tag'] ) ? 'edgtf-ll-tag-set' : '';
		$classes[] = ! empty( $params['amenity'] ) ? 'edgtf-ll-amenity-set' : '';
		
		return implode( ' ', $classes );
	}
	
	/**
	 * Generates holder inner classes
	 *
	 * @param $params
	 *
	 * @return string
	 */
	public function getHolderInnerClasses( $params ) {
		$classes = array();
		
		$classes[] = 'edgtf-outer-space';
		
		$classes[] = ! empty( $params['enable_map'] ) && $params['enable_map'] == 'yes' ? 'edgtf-ml-inner' : '';
		
		return implode( ' ', $classes );
	}
	
	/**
	 * Generates article classes
	 *
	 *
	 * @return string
	 */
	public function getArticleClasses( $params ) {
		$classes = array();
		
		$classes[] = 'edgtf-item-space';
		
		$item_is_featured = get_post_meta( get_the_ID(), 'edgtf_listing_is_featured_meta', true );
		if ( $item_is_featured === 'yes' ) {
			$classes[] = 'edgtf-featured-item';
		}
		
		$article_classes = get_post_class( $classes );
		
		return implode( ' ', $article_classes );
	}
	
	/**
	 * Generates image size
	 *
	 * @param $params
	 *
	 * @return string
	 */
	public function getImageSize( $params ) {
		$thumb_size = 'full';
		
		if ( ! empty( $params['image_proportions'] ) && $params['type'] == 'gallery' ) {
			$image_size = $params['image_proportions'];
			
			switch ( $image_size ) {
				case 'landscape':
					$thumb_size = 'urbango_edge_image_landscape';
					break;
				case 'portrait':
					$thumb_size = 'urbango_edge_image_portrait';
					break;
				case 'square':
					$thumb_size = 'urbango_edge_image_square';
					break;
				case 'thumbnail':
					$thumb_size = 'thumbnail';
					break;
				case 'medium':
					$thumb_size = 'medium';
					break;
				case 'large':
					$thumb_size = 'large';
					break;
				case 'full':
					$thumb_size = 'full';
					break;
			}
		}
		
		return $thumb_size;
	}
	
	public function getLoadMoreStyles( $params ) {
		$styles = array();
		
		if ( ! empty( $params['load_more_top_margin'] ) ) {
			$margin = $params['load_more_top_margin'];
			
			if ( urbango_edge_string_ends_with( $margin, '%' ) || urbango_edge_string_ends_with( $margin, 'px' ) ) {
				$styles[] = 'margin-top: ' . $margin;
			} else {
				$styles[] = 'margin-top: ' . urbango_edge_filter_px( $margin ) . 'px';
			}
		}
		
		return implode( ';', $styles );
	}
	
	/**
	 * Filter listing items by ID or Title
	 *
	 * @param $query
	 *
	 * @return array
	 */
	public function listingIdAutocompleteSuggester( $query ) {
		global $wpdb;
		$listing_id      = (int) $query;
		$post_meta_infos = $wpdb->get_results( $wpdb->prepare( "SELECT ID AS id, post_title AS title
					FROM {$wpdb->posts} 
					WHERE post_type = 'listing-item' AND ( ID = '%d' OR post_title LIKE '%%%s%%' )", $listing_id > 0 ? $listing_id : - 1, stripslashes( $query ), stripslashes( $query ) ), ARRAY_A );
		
		$results = array();
		if ( is_array( $post_meta_infos ) && ! empty( $post_meta_infos ) ) {
			foreach ( $post_meta_infos as $value ) {
				$data          = array();
				$data['value'] = $value['id'];
				$data['label'] = esc_html__( 'Id', 'urbango-listing' ) . ': ' . $value['id'] . ( ( strlen( $value['title'] ) > 0 ) ? ' - ' . esc_html__( 'Title', 'urbango-listing' ) . ': ' . $value['title'] : '' );
				$results[]     = $data;
			}
		}
		
		return $results;
	}
	
	/**
	 * Find listing by id
	 * @since 4.4
	 *
	 * @param $query
	 *
	 * @return bool|array
	 */
	public function listingIdAutocompleteRender( $query ) {
		$query = trim( $query['value'] ); // get value from requested
		if ( ! empty( $query ) ) {
			// get listing
			$listing = get_post( (int) $query );
			if ( ! is_wp_error( $listing ) ) {
				
				$listing_id    = $listing->ID;
				$listing_title = $listing->post_title;
				
				$listing_title_display = '';
				if ( ! empty( $listing_title ) ) {
					$listing_title_display = ' - ' . esc_html__( 'Title', 'urbango-listing' ) . ': ' . $listing_title;
				}
				
				$listing_id_display = esc_html__( 'Id', 'urbango-listing' ) . ': ' . $listing_id;
				
				$data          = array();
				$data['value'] = $listing_id;
				$data['label'] = $listing_id_display . $listing_title_display;
				
				return ! empty( $data ) ? $data : false;
			}
			
			return false;
		}
		
		return false;
	}
}
