<?php

namespace UrbanGoListing\CPT\Shortcodes\Listing;

use UrbanGoListing\Lib;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

class CategoryTabs implements Lib\ShortcodeInterface {
	private $base;
	
	public function __construct() {
		$this->base = 'edgtf_category_tabs';
		
		add_action( 'vc_before_init', array( $this, 'vcMap' ) );
	}
	
	public function getBase() {
		return $this->base;
	}
	
	public function vcMap() {
		if ( function_exists( 'vc_map' ) ) {
			vc_map( array(
					'name'     => esc_html__( 'Category Tabs', 'urbango-listing' ),
					'base'     => $this->getBase(),
					'category' => esc_html__( 'by URBANGO LISTING', 'urbango-listing' ),
					'icon'     => 'icon-wpb-category-tabs extended-custom-listing-icon',
					'params'   => array(
						array(
							'type'        => 'textfield',
							'param_name'  => 'number_of_items',
							'heading'     => esc_html__( 'Number of Items', 'urbango-listing' ),
							'description' => esc_html__( 'Set number of items for your tabs', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'category',
							'heading'     => esc_html__( 'Category', 'urbango-listing' ),
							'value'       => array_flip( urbango_listing_get_cpt_taxonomy_items( 'category' ) ),
							'save_always' => true
						),
						array(
							'type'        => 'textfield',
							'param_name'  => 'category_slugs',
							'heading'     => esc_html__( 'Category Slugs', 'urbango-listing' ),
							'description' => esc_html__( 'Enter the category slugs you want to display, separated by commas (leave empty for showing all categories)', 'urbango-listing' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'order_by',
							'heading'     => esc_html__( 'Order By', 'urbango-listing' ),
							'value'       => array_flip( urbango_edge_get_query_order_by_array() )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'order',
							'heading'     => esc_html__( 'Order', 'urbango-listing' ),
							'value'       => array_flip( urbango_edge_get_query_order_array() )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'term_icon_type',
							'heading'    => esc_html__( 'Category Icon Type', 'urbango-listing' ),
							'value'      => array(
								esc_html__( 'Icon', 'urbango-listing' )     => 'icon',
								esc_html__( 'SVG Icon', 'urbango-listing' ) => 'svg-icon',
								esc_html__( 'No Icon', 'urbango-listing' )  => 'disable'
							),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'title_tag',
							'heading'    => esc_html__( 'Title Tag', 'urbango-listing' ),
							'value'      => array_flip( urbango_edge_get_title_tag( true, array( 'p' => 'p' ) ) ),
							'dependency' => array( 'element' => 'enable_title', 'value' => array( 'yes' ) ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'colorpicker',
							'param_name' => 'title_color',
							'heading'    => esc_html__( 'Title Color', 'urbango-listing' ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'textfield',
							'param_name' => 'title_font_size',
							'heading'    => esc_html__( 'Title Font Size (px)', 'urbango-listing' ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						),
						array(
							'type'       => 'textfield',
							'param_name' => 'title_top_margin',
							'heading'    => esc_html__( 'Title Top Margin (px)', 'urbango-listing' ),
							'group'      => esc_html__( 'Content Layout', 'urbango-listing' )
						)
					)
				)
			);
		}
	}
	
	public function render( $atts, $content = null ) {
		$args   = array(
			'number_of_items'  => '9',
			'category'         => '',
			'category_slugs'   => '',
			'order_by'         => 'date',
			'order'            => 'ASC',
			'term_icon_type'   => 'icon',
			'title_tag'        => 'p',
			'title_color'      => '',
			'title_font_size'  => '',
			'title_top_margin' => ''
		);
		$params = shortcode_atts( $args, $atts );
		
		/***
		 * @params query_results
		 * @params holder_classes
		 */
		$additional_params = array();
		
		$additional_params['query_results'] = $this->getQueryArray( $params );
		
		$params['title_styles'] = $this->getTitleStyles( $params );
		
		$html = urbango_listing_get_module_template_part( 'cpt-listing', 'category-tabs', 'holder', '', $params, $additional_params, false );
		
		return $html;
	}
	
	/**
	 * Generates query results
	 *
	 * @param $params
	 *
	 * @return array
	 */
	public function getQueryArray( $params ) {
		$query_args = array(
			'taxonomy' => 'listing-category',
			'number'   => $params['number_of_items'],
			'orderby'  => $params['order_by'],
			'order'    => $params['order']
		);
		
		if ( $params['order_by'] === 'menu_order' ) {
			$query_args['orderby']    = 'edgtf_listing_category_order_meta';
			$query_args['meta_query'] = array(
				'key'     => 'edgtf_listing_category_order_meta',
				'compare' => 'EXISTS'
			);
		}
		
		$category = trim( $params['category'] );
		if ( ! empty( $category ) ) {
			$query_args['include'] = array( intval( $category ) );
		}
		
		$category_slugs = trim( $params['category_slugs'] );
		if ( ! empty( $category_slugs ) && empty( $category ) ) {
			$query_args['slug'] = explode( ',', str_replace( ' ', '', $category_slugs ) );
		}
		
		$query_array = get_terms( $query_args );
		
		return $query_array;
	}
	
	private function getTitleStyles( $params ) {
		$styles = array();
		
		if ( ! empty( $params['title_color'] ) ) {
			$styles[] = 'color: ' . $params['title_color'];
		}
		
		if ( ! empty( $params['title_font_size'] ) ) {
			$styles[] = 'font-size: ' . urbango_edge_filter_px( $params['title_font_size'] ) . 'px';
		}
		
		if ( $params['title_top_margin'] !== '' ) {
			$styles[] = 'margin-top: ' . urbango_edge_filter_px( $params['title_top_margin'] ) . 'px';
		}
		
		return implode( ';', $styles );
	}
}
