<?php

namespace UrbanGoListing\CPT\Listing;

use UrbanGoListing\PostTypes\PostTypesInterface;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Class ListingRegister
 * @package UrbanGoListing\CPT\Listing
 */
class ListingRegister implements PostTypesInterface {
	
	/**
	 * Singleton variables
	 */
	private $base;
	
	public function __construct() {
		$this->base = 'listing-item';
		
		add_filter( 'archive_template', array( $this, 'registerArchiveTemplate' ) );
		add_filter( 'single_template', array( $this, 'registerSingleTemplate' ) );
	}
	
	/**
	 * @return string
	 */
	public function getBase() {
		return $this->base;
	}
	
	/**
	 * Registers archive template if one doesn't exists in theme.
	 * Hooked to archive_template filter
	 *
	 * @param $archive string current template
	 *
	 * @return string string changed template
	 */
	public function registerArchiveTemplate( $archive ) {
		global $post;
		
		if ( is_author() ) {
			return URBANGO_LISTING_CPT_PATH . '/' . str_replace( '-item', '', $this->getBase() ) . '/templates/archive-author.php';
		}
		
		if ( isset( $post ) && $post->post_type == $this->getBase() ) {
			if ( ! file_exists( get_template_directory() . '/archive-' . $this->getBase() . '.php' ) ) {
				return URBANGO_LISTING_CPT_PATH . '/' . str_replace( '-item', '', $this->getBase() ) . '/templates/archive-' . $this->getBase() . '.php';
			}
		}
		
		return $archive;
	}
	
	/**
	 * Registers single template if one doesn't exists in theme.
	 * Hooked to single_template filter
	 *
	 * @param $single string current template
	 *
	 * @return string string changed template
	 */
	public function registerSingleTemplate( $single ) {
		global $post;
		
		if ( ! empty( $post ) && $post->post_type == $this->getBase() ) {
			if ( ! file_exists( get_template_directory() . '/single-' . $this->getBase() . '.php' ) ) {
				return URBANGO_LISTING_CPT_PATH . '/' . str_replace( '-item', '', $this->getBase() ) . '/templates/single-' . $this->getBase() . '.php';
			}
		}
		
		return $single;
	}
	
	/**
	 * Registers custom post type and taxonomies
	 */
	public function register() {
		$this->registerPostType();
		$this->registerTaxonomies();
	}
	
	/**
	 * Registers custom post type with WordPress
	 */
	private function registerPostType() {
		$menuPosition = 32;
		$menuIcon     = 'dashicons-feedback';
		$slugMeta     = urbango_edge_options()->getOptionValue( 'listing_single_slug' );
		$slug         = isset( $slugMeta ) && ! empty( $slugMeta ) ? $slugMeta : $this->getBase();
		
		register_post_type( $this->getBase(),
			array(
				'labels'        => array(
					'name'          => esc_html__( 'UrbanGo Listing', 'urbango-listing' ),
					'singular_name' => esc_html__( 'Listing Item', 'urbango-listing' ),
					'add_item'      => esc_html__( 'New Listing Item', 'urbango-listing' ),
					'add_new_item'  => esc_html__( 'Add New Listing Item', 'urbango-listing' ),
					'edit_item'     => esc_html__( 'Edit Listing Item', 'urbango-listing' )
				),
				'public'        => true,
				'has_archive'   => true,
				'rewrite'       => array( 'slug' => $slug ),
				'menu_position' => $menuPosition,
				'show_in_menu'  => true,
				'show_ui'       => true,
				'supports'      => array(
					'author',
					'revisions',
					'title',
					'editor',
					'thumbnail',
					'excerpt',
					'page-attributes',
					'comments'
				),
				'menu_icon'     => $menuIcon
			)
		);
	}
	
	/**
	 * Registers custom taxonomies for CPT with WordPress
	 */
	private function registerTaxonomies() {
		$cptTaxonomies = array(
			'category' => array(
				'key'        => 'listing-category',
				'name'       => esc_html__( 'Categories', 'urbango-listing' ),
				'label'      => esc_html__( 'Category', 'urbango-listing' ),
				'show_ui'    => null,
				'show_admin' => true
			),
			'tag'      => array(
				'key'        => 'listing-tag',
				'name'       => esc_html__( 'Tags', 'urbango-listing' ),
				'label'      => esc_html__( 'Tag', 'urbango-listing' ),
				'show_ui'    => null,
				'show_admin' => false
			),
			'location' => array(
				'key'        => 'listing-location',
				'name'       => esc_html__( 'Locations', 'urbango-listing' ),
				'label'      => esc_html__( 'Location', 'urbango-listing' ),
				'show_ui'    => null,
				'show_admin' => true
			),
			'amenity'  => array(
				'key'        => 'listing-amenity',
				'name'       => esc_html__( 'Amenities', 'urbango-listing' ),
				'label'      => esc_html__( 'Amenity', 'urbango-listing' ),
				'show_ui'    => null,
				'show_admin' => false
			),
			'reviews-criteria'  => array(
				'key'        => 'listing-reviews-criteria',
				'name'       => esc_html__( 'Reviews Criteria', 'urbango-listing' ),
				'label'      => esc_html__( 'Reviews Criteria', 'urbango-listing' ),
				'show_ui'    => false,
				'show_admin' => false
			)
		);
		
		foreach ( $cptTaxonomies as $key => $value ) {
			$labels = array(
				'name'              => $value['name'],
				'singular_name'     => $value['label'],
				'search_items'      => sprintf( esc_html__( 'Search %s', 'urbango-listing' ), $value['name'] ),
				'all_items'         => sprintf( esc_html__( 'All %s', 'urbango-listing' ), $value['name'] ),
				'parent_item'       => sprintf( esc_html__( 'Parent %s', 'urbango-listing' ), $value['label'] ),
				'parent_item_colon' => sprintf( esc_html__( 'Parent %s:', 'urbango-listing' ), $value['label'] ),
				'edit_item'         => sprintf( esc_html__( 'Edit %s', 'urbango-listing' ), $value['label'] ),
				'update_item'       => sprintf( esc_html__( 'Update %s', 'urbango-listing' ), $value['label'] ),
				'add_new_item'      => sprintf( esc_html__( 'Add New %s', 'urbango-listing' ), $value['label'] ),
				'new_item_name'     => sprintf( esc_html__( 'New %s Name', 'urbango-listing' ), $value['label'] ),
				'menu_name'         => $value['name']
			);
			
			register_taxonomy( $value['key'], array( $this->getBase() ), array(
				'hierarchical'       => true,
				'labels'             => $labels,
				'show_ui'            => true,
				'show_in_quick_edit' => $value['show_ui'],
				'meta_box_cb'        => $value['show_ui'],
				'query_var'          => true,
				'show_admin_column'  => $value['show_admin'],
				'rewrite'            => array( 'slug' => $value['key'] )
			) );
		}
	}
}
