<?php

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

if ( ! function_exists( 'urbango_listing_add_google_maps_extensions' ) ) {
	function urbango_listing_add_google_maps_extensions( $extensions ) {
		$items      = array(
			'geometry',
			'places'
		);
		$extensions = array_unique( array_merge( $extensions, $items ) );

		return $extensions;
	}

	add_filter( 'urbango_edge_filter_google_maps_extensions_array', 'urbango_listing_add_google_maps_extensions' );
}

if ( ! function_exists( 'urbango_listing_enable_google_maps_in_admin' ) ) {
	function urbango_listing_enable_google_maps_in_admin() {
		return true;
	}

	add_action( 'urbango_edge_filter_google_maps_in_backend', 'urbango_listing_enable_google_maps_in_admin' );
}

if ( ! function_exists( 'urbango_listing_set_global_map_variables' ) ) {
	/**
	 * Function for setting global map variables
	 */
	function urbango_listing_set_global_map_variables() {
		$global_map_variables = array();

		$global_map_variables['mapStyle']          = json_decode( urbango_edge_options()->getOptionValue( 'listing_map_style' ) );
		$global_map_variables['scrollable']        = urbango_edge_options()->getOptionValue( 'listing_maps_scrollable' ) == 'yes' ? true : false;
		$global_map_variables['draggable']         = urbango_edge_options()->getOptionValue( 'listing_maps_draggable' ) == 'yes' ? true : false;
		$global_map_variables['streetViewControl'] = urbango_edge_options()->getOptionValue( 'listing_maps_street_view_control' ) == 'yes' ? true : false;
		$global_map_variables['zoomControl']       = urbango_edge_options()->getOptionValue( 'listing_maps_zoom_control' ) == 'yes' ? true : false;
		$global_map_variables['mapTypeControl']    = urbango_edge_options()->getOptionValue( 'listing_maps_type_control' ) == 'yes' ? true : false;

		$global_map_variables = apply_filters( 'urbango_listing_filter_js_global_map_variables', $global_map_variables );

		wp_localize_script( 'urbango-edge-modules', 'edgtfMapsVars', array(
			'global' => $global_map_variables
		) );
	}

	add_action( 'wp_enqueue_scripts', 'urbango_listing_set_global_map_variables', 20 );
}

/* MULTIPLE MAP FUNCTIONS - START */
if ( ! function_exists( 'urbango_listing_set_multiple_map_variables' ) ) {
	/**
	 * Function for setting single map variables
	 *
	 * @param $query - $query is used just for multiple type. $query is Wp_Query args object containing listing items which should be presented on map
	 * @param $return - whether map object should be returned (for ajax call) or passed to localize script
	 *
	 * @return array - array with addresses parameters
	 */
	function urbango_listing_set_multiple_map_variables( $query = '', $return = false ) {
		$map_variables = array();
		$items         = urbango_listing_get_cpt_items( 'listing-item', $query );

		if ( ! empty( $items ) ) {
			foreach ( $items as $id => $title ) {
				$map_variables['addresses'][] = urbango_listing_generate_listing_map_params( $id );
			}
		}

		if ( $return ) {
			return $map_variables;
		}

		wp_localize_script( 'urbango-edge-modules', 'edgtfMultipleMapVars', array(
			'multiple' => $map_variables
		) );
	}
}

if ( ! function_exists( 'urbango_listing_get_listing_multiple_map' ) ) {
	/**
	 * Function that renders map holder for multiple listing item
	 *
	 * @param $query - $query is used just for multiple type. $query is Wp_Query object containing listing items which should be presented on map
	 *
	 * @return string
	 */
	function urbango_listing_get_listing_multiple_map( $query = '' ) {
		urbango_listing_set_multiple_map_variables( $query );

		$google_maps_api_key = urbango_edge_options()->getOptionValue( 'google_maps_api_key' );
		if ( ! empty( $google_maps_api_key ) ) {
			$html = '<div id="edgtf-listing-multiple-map-holder"></div>';
		} else {
			$html = '<p id="edgtf-listing-multiple-map-notice">' . esc_html__( 'In order for the map functionality to be enabled please input the Google Map API key in the General section of the UrbanGo Options', 'urbango-listing' ) . '</p>';
		}

		do_action( 'urbango_listing_action_after_listing_map' );

		return $html;
	}
}

/* MULTIPLE MAP FUNCTIONS - START */

/* MAP ITEMS FUNCTIONS START - */
if ( ! function_exists( 'urbango_listing_marker_info_template' ) ) {
	/**
	 * Template with placeholders for marker info window
	 *
	 * uses underscore templates
	 */
	function urbango_listing_marker_info_template() {

		$html = '<script type="text/template" class="edgtf-info-window-template">
				<div class="edgtf-info-window">
					<div class="edgtf-info-window-inner">
						<a itemprop="url" class="edgtf-info-window-link" href="<%= itemUrl %>"></a>
						<% if ( featuredImage ) { %>
							<div class="edgtf-info-window-image">
								<img itemprop="image" src="<%= featuredImage[0] %>" alt="<%= title %>" width="<%= featuredImage[1] %>" height="<%= featuredImage[2] %>">
							</div>
						<% } %>
						<div class="edgtf-info-window-details">
							<h6 itemprop="name" class="edgtf-info-window-title"><%= title %></h6>
							<p class="edgtf-info-window-location"><%= address %></p>
						</div>
					</div>
				</div>
			</script>';

		if ( function_exists( 'urbango_edge_get_module_part' ) ) {
			print urbango_edge_get_module_part( $html );
		}
	}

	add_action( 'urbango_listing_action_after_listing_map', 'urbango_listing_marker_info_template' );
}

if ( ! function_exists( 'urbango_listing_marker_template' ) ) {
	/**
	 * Template with placeholders for marker
	 */
	function urbango_listing_marker_template() {

		$html = '<script type="text/template" class="edgtf-marker-template">
				<div class="edgtf-map-marker">
					<div class="edgtf-map-marker-inner">
						<%= termIcon %>
						<svg class="edgtf-map-marker-pin" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="37.875px" height="50.75px" viewBox="0 0 37.875 50.75" enable-background="new 0 0 37.875 50.75" xml:space="preserve"><g><path fill="#EF4960" d="M0,18.938C0,29.396,17.746,50.75,18.938,50.75V0C8.479,0,0,8.479,0,18.938z"/><path fill="#DC4458" d="M37.875,18.938C37.875,8.479,29.396,0,18.938,0v50.75C20.129,50.75,37.875,29.396,37.875,18.938z"/></g><circle fill="#FFFFFF" cx="18.938" cy="19.188" r="14.813"/></svg>
					</div>
				</div>
			</script>';

		if ( function_exists( 'urbango_edge_get_module_part' ) ) {
			print urbango_edge_get_module_part( $html );
		}
	}

	add_action( 'urbango_listing_action_after_listing_map', 'urbango_listing_marker_template' );
}

/* MAP ITEMS FUNCTIONS - END */

/* HELPER FUNCTIONS - START */

if ( ! function_exists( 'urbango_listing_generate_listing_map_params' ) ) {
	function urbango_listing_generate_listing_map_params( $item_id ) {
		$map_params = array();

		// Get listing image
		$image_id = get_post_thumbnail_id( $item_id );
		$image    = wp_get_attachment_image_src( $image_id );

		// Get address params
		$address_array = urbango_listing_get_address_params( $item_id );

		// Get item location
		if ( $address_array['address'] === '' && $address_array['address_lat'] === '' && $address_array['address_long'] === '' ) {
			$map_params['location'] = null;
		} else {
			$map_params['location'] = array(
				'address'   => $address_array['address'],
				'latitude'  => $address_array['address_lat'],
				'longitude' => $address_array['address_long']
			);
		}

		// Get terms icon
		$term_ids = wp_get_post_terms( $item_id, 'listing-category' );
		$term_id  = '';
		if ( ! empty( $term_ids ) ) {
			foreach ( $term_ids as $term ) {
				if ( $term->parent === 0 ) {
					$term_id = $term->term_id;
					break;
				}
			}
		}
		$term_icon = '';

		if ( ! empty( $term_id ) ) {
			$term_custom_icon = get_term_meta( $term_id, 'edgtf_listing_category_icon_meta', true );
			$term_svg_icon    = get_term_meta( $term_id, 'edgtf_listing_category_svg_icon_meta', true );

			if ( ! empty( $term_svg_icon ) ) {
				$term_icon = '<div class="edgtf-map-marker-term-icon">' . stripslashes( preg_replace( '/<script\b[^>]*>(.*?)<\/script>/is', '', $term_svg_icon ) ) . '</div>';
			} else if ( ! empty( $term_custom_icon ) ) {
				$term_icon = '<div class="edgtf-map-marker-term-icon">' . wp_get_attachment_image( $term_custom_icon, 'full' ) . '</div>';
			}
		}

		$map_params['title']         = get_the_title( $item_id );
		$map_params['itemId']        = $item_id;
		$map_params['featuredImage'] = $image;
		$map_params['termIcon']      = $term_icon;
		$map_params['itemUrl']       = get_the_permalink( $item_id );

		return $map_params;
	}
}

if ( ! function_exists( 'urbango_listing_get_address_params' ) ) {
	/**
	 * Function that set up address params
	 *
	 * @param $id - id of current post
	 *
	 * @return array
	 */
	function urbango_listing_get_address_params( $id ) {
		$address_array  = array();
		$address_string = get_post_meta( $id, 'edgtf_listing_single_full_address_meta', true );
		$address_lat    = get_post_meta( $id, 'edgtf_listing_single_full_address_latitude_meta', true );
		$address_long   = get_post_meta( $id, 'edgtf_listing_single_full_address_longitude_meta', true );

		$address_array['address']      = $address_string !== '' ? $address_string : '';
		$address_array['address_lat']  = $address_lat !== '' ? $address_lat : '';
		$address_array['address_long'] = $address_long !== '' ? $address_long : '';

		return $address_array;
	}
}

/* HELPER FUNCTIONS - END */