<?php

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

if ( ! function_exists( 'urbango_listing_version_classes' ) ) {
	/**
	 * Adds plugins version classes to the body
	 *
	 * @param $classes
	 *
	 * @return array
	 */
	function urbango_listing_version_classes( $classes ) {
		$classes[] = 'urbango-listing-' . URBANGO_LISTING_VERSION;

		return $classes;
	}

	add_filter( 'body_class', 'urbango_listing_version_classes' );
}

if ( ! function_exists( 'urbango_listing_get_module_template_part' ) ) {
	/**
	 * Loads module template part.
	 *
	 * @param string $module name of the module
	 * @param string $module_name name of the module folder
	 * @param string $template name of the template to load
	 * @param string $slug
	 * @param array $params array of parameters to pass to template
	 * @param array $additional_params array of additional parameters to pass to template
	 * @param bool $echo - print or return template content
	 *
	 * @return string/html
	 */
	function urbango_listing_get_module_template_part( $module, $module_name, $template, $slug = '', $params = array(), $additional_params = array(), $echo = true ) {
		$available_characters = '/[^A-Za-z0-9\_\-\/]/';

		if ( is_scalar( $module ) ) {
			$module = preg_replace( $available_characters, '', $module );
		} else {
			$module = '';
		}

		if ( is_scalar( $template ) ) {
			$template = preg_replace( $available_characters, '', $template );
		} else {
			$template = '';
		}

		if ( is_scalar( $slug ) ) {
			$slug = preg_replace( $available_characters, '', $slug );
		} else {
			$slug = '';
		}

		if ( is_scalar( $module_name ) ) {
			$module_name = preg_replace( $available_characters, '', $module_name );
		} else {
			$module_name = '';
		}
		
		//HTML Content from template
		$html        = '';
		$module_root = '';

		switch ( $module ) {
			case 'modules':
				$module_root = URBANGO_LISTING_MODULES_PATH . '/';
				break;
			case 'cpt':
				$module_root = URBANGO_LISTING_CPT_PATH . '/';
				break;
			case 'cpt-listing':
				$module_root = URBANGO_LISTING_CPT_PATH . '/listing/shortcodes/';
				$module_name = $module_name . '/templates';
				break;
			case 'cpt-listing-profile':
				$module_root = URBANGO_LISTING_CPT_PATH . '/' . $module_name . '/profile/';
				$module_name = '/templates';
				break;
			case 'cpt-package':
				$module_root = URBANGO_LISTING_CPT_PATH . '/package/shortcodes/';
				$module_name = $module_name . '/templates';
				break;
			case 'cpt-package-profile':
				$module_root = URBANGO_LISTING_CPT_PATH . '/' . $module_name . '/profile/';
				$module_name = '/templates';
				break;
			case 'cpt-report-profile':
				$module_root = URBANGO_LISTING_CPT_PATH . '/' . $module_name . '/profile/';
				$module_name = '/templates';
				break;
			case 'cpt-claim-profile':
				$module_root = URBANGO_LISTING_CPT_PATH . '/' . $module_name . '/profile/';
				$module_name = '/templates';
				break;
		}

		$temp = $module_root . $module_name . '/' . $template;

		if ( is_array( $params ) && count( $params ) ) {
			extract( $params, EXTR_SKIP ); // @codingStandardsIgnoreLine
		}

		if ( is_array( $additional_params ) && count( $additional_params ) ) {
			extract( $additional_params, EXTR_SKIP ); // @codingStandardsIgnoreLine
		}

		$template = '';

		if ( ! empty( $temp ) ) {
			if ( ! empty( $slug ) ) {
				$template = "{$temp}-{$slug}.php";

				if ( ! file_exists( $template ) ) {
					$template = $temp . '.php';
				}
			} else {
				$template = $temp . '.php';
			}
		}

		if ( ! empty( $template ) ) {
			ob_start();
			include( $template );
			$html = ob_get_clean();
		}

		if ( $echo ) {
			if ( function_exists( 'urbango_edge_get_module_part' ) ) {
				print urbango_edge_get_module_part( $html );
			}
		} else {
			return $html;
		}
	}
}

if ( ! function_exists( 'urbango_listing_get_ajax_status' ) ) {
	/**
	 * Function that return status from ajax functions
	 */
	function urbango_listing_get_ajax_status( $status, $message, $data = null, $redirect = '' ) {
		$response = array(
			'status'   => $status,
			'message'  => $message,
			'data'     => $data,
			'redirect' => $redirect
		);

		$output = json_encode( $response );

		exit( $output );
	}
}

if ( ! function_exists( 'urbango_listing_check_is_user_admin' ) ) {
	/**
	 * Function that check is current user admin
	 */
	function urbango_listing_check_is_user_admin() {
		return is_user_logged_in() && current_user_can( 'edit_theme_options' );
	}
}

if ( ! function_exists( 'urbango_listing_get_cpt_items' ) ) {
	function urbango_listing_get_cpt_items( $cpt_slug, $args = array(), $first_empty = false ) {
		$items      = array();
		$query_args = array(
			'post_status'    => 'publish',
			'post_type'      => $cpt_slug,
			'posts_per_page' => '-1',
			'fields'         => 'ids'
		);

		if ( ! empty( $args ) ) {
			foreach ( $args as $key => $value ) {
				if ( ! empty( $value ) ) {
					$query_args[ $key ] = $value;
				}
			}
		}

		$cpt_items = new \WP_Query( $query_args );

		if ( $cpt_items->have_posts() ) {

			if ( $first_empty ) {
				$items[''] = esc_html__( 'Default', 'urbango-listing' );
			}

			foreach ( $cpt_items->posts as $id ):
				$items[ $id ] = get_the_title( $id );
			endforeach;
		}

		wp_reset_postdata();

		return $items;
	}
}

if ( ! function_exists( 'urbango_listing_get_cpt_taxonomy_items' ) ) {
	function urbango_listing_get_cpt_taxonomy_items( $taxonomy_slug, $first_empty = true, $taxonomy_cpt = 'listing', $only_parent = false ) {
		$items = array();

		if ( ! empty( $taxonomy_slug ) ) {
			$terms_args = array(
				'hide_empty' => false
			);

			if ( $only_parent ) {
				$terms_args['parent'] = 0;
			}
			$terms = get_terms( $terms_args );

			if ( is_array( $terms ) && ! empty( $terms ) ) {

				if ( $first_empty ) {
					$items[''] = esc_html__( 'Default', 'urbango-listing' );
				}

				foreach ( $terms as $term ) {
					if ( $term->taxonomy == $taxonomy_cpt . '-' . $taxonomy_slug ) {
						// use only with specific taxonomy (doesn't work like argument in get_terms() because our meta boxes are init early)
						$items[ $term->term_id ] = $term->name;
					}
				}
			}
		}

		return $items;
	}
}

if ( ! function_exists( 'urbango_listing_get_cpt_taxonomy_items_name_by_ids' ) ) {
	function urbango_listing_get_cpt_taxonomy_items_name_by_ids( $term_ids ) {
		if ( ! empty ( $term_ids ) ) {
			$ids       = explode( ',', $term_ids );
			$termsName = array();

			foreach ( $ids as $id ) {
				$term = get_term( $id );

				if ( ! empty( $term ) ) {
					$termsName[] = $term->name;
				}
			}

			return implode( ', ', $termsName );
		} else {
			return '';
		}
	}
}

if ( ! function_exists( 'urbango_listing_get_cpt_taxonomy_name_by_id' ) ) {
	function urbango_listing_get_cpt_taxonomy_name_by_id( $term_id ) {
		if ( ! empty ( $term_id ) ) {
			$term = get_term( $term_id );

			return ! empty( $term ) ? $term->name : '';
		} else {
			return '';
		}
	}
}

if ( ! function_exists( 'urbango_listing_get_all_users' ) ) {
	/**
	 * Returns array of users
	 *
	 * @param $first_empty bool
	 * @param $additional_args array - set additional args
	 *
	 * @return array
	 */
	function urbango_listing_get_all_users( $first_empty = true, $additional_args = array() ) {
		$options   = array();
		$user_args = array(
			'fields'  => array( 'ID', 'display_name' ),
			'orderby' => 'display_name'
		);

		if ( ! empty( $additional_args ) && is_array( $additional_args ) ) {
			$user_args = array_merge( $user_args, $additional_args );
		}

		$users = get_users( $user_args );

		if ( ! empty( $users ) ) {
			if ( $first_empty ) {
				$options[''] = esc_html__( 'Default', 'urbango-listing' );
			}

			foreach ( $users as $user ) {
				$options[ $user->ID ] = $user->display_name;
			}
		}

		return $options;
	}
}

if ( ! function_exists( 'urbango_listing_get_listing_price_range' ) ) {
	/**
	 * Returns array of price range
	 *
	 * @return array
	 */
	function urbango_listing_get_listing_price_range() {
		$options = array();

		$options['0'] = esc_html__( 'No Price Range', 'urbango-listing' );
		$options['1'] = esc_html__( 'Low', 'urbango-listing' );
		$options['2'] = esc_html__( 'Cheap', 'urbango-listing' );
		$options['3'] = esc_html__( 'Medium', 'urbango-listing' );
		$options['4'] = esc_html__( 'Expensive', 'urbango-listing' );
		$options['5'] = esc_html__( 'Luxury', 'urbango-listing' );

		return $options;
	}
}

if ( ! function_exists( 'urbango_listing_get_date_time_zone' ) ) {
	/**
	 * Returns array of date time zone
	 *
	 * @param $first_empty bool
	 *
	 * @return array
	 */
	function urbango_listing_get_date_time_zone( $first_empty = true ) {
		$options = array();

		if ( class_exists( 'DateTimeZone' ) ) {
			$options = DateTimeZone::listIdentifiers( DateTimeZone::ALL );
		}

		if ( $first_empty ) {
			array_unshift( $options, esc_html__( 'Default', 'urbango-listing' ) );
		}

		return $options;
	}
}

if ( ! function_exists( 'urbango_listing_get_date_time_zone_name_by_id' ) ) {
	/**
	 * Returns name of the date time zone
	 *
	 * @param $id int time zone ID
	 *
	 * @return array
	 */
	function urbango_listing_get_date_time_zone_name_by_id( $id ) {
		$time_zone_name = '';
		$time_zones     = urbango_listing_get_date_time_zone( false );

		if ( ! empty( $time_zones ) && ! empty( $id ) && isset( $time_zones[ $id - 1 ] ) ) {
			$time_zone_name = $time_zones[ $id - 1 ]; // -1 is set because of Default option value that move all items index for 1
		}

		return $time_zone_name;
	}
}

if ( ! function_exists( 'urbango_listing_get_weekend_days' ) ) {
	/**
	 * Returns array of days in weekend
	 *
	 * @return array
	 */
	function urbango_listing_get_weekend_days() {
		$options = array();

		$options['monday']    = esc_html__( 'Monday', 'urbango-listing' );
		$options['tuesday']   = esc_html__( 'Tuesday', 'urbango-listing' );
		$options['wednesday'] = esc_html__( 'Wednesday', 'urbango-listing' );
		$options['thursday']  = esc_html__( 'Thursday', 'urbango-listing' );
		$options['friday']    = esc_html__( 'Friday', 'urbango-listing' );
		$options['saturday']  = esc_html__( 'Saturday', 'urbango-listing' );
		$options['sunday']    = esc_html__( 'Sunday', 'urbango-listing' );

		return $options;
	}
}

if ( ! function_exists( 'urbango_listing_get_hours_range' ) ) {
	/**
	 * Returns array of hours range
	 *
	 * @param $params array - config param
	 *
	 * @return array
	 */
	function urbango_listing_get_hours_range( $params = array() ) {
		// Default values
		$start  = 0;
		$end    = 86400;
		$step   = 1800;
		$format = 'g:ia';

		extract( $params, EXTR_SKIP ); // @codingStandardsIgnoreLine

		$times = array();

		foreach ( range( $start, $end, $step ) as $timestamp ) {
			$hour = gmdate( $format, $timestamp );

			$times[ $hour ] = $hour;
		}

		return $times;
	}
}

if ( ! function_exists( 'urbango_listing_get_people_range' ) ) {
	/**
	 * Returns array of peoples
	 *
	 * @return array
	 */
	function urbango_listing_get_people_range() {
		$options = array();

		for ( $i = 1; $i <= 20; $i ++ ) {
			$options[ $i ] = sprintf( '%s %s', $i, $i === 1 ? esc_html__( 'Person', 'urbango-listing' ) : esc_html__( 'People', 'urbango-listing' ) );
		}

		return apply_filters( 'urbango_listing_filter_get_people_range', $options );
	}
}

if ( ! function_exists( 'urbango_listing_get_featured_mark' ) ) {
	/**
	 * Returns featured svg mark
	 *
	 * @return html
	 */
	function urbango_listing_get_featured_mark() {
		$html = '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="11.938px" height="15.181px" viewBox="0 0 11.938 15.181" enable-background="new 0 0 11.938 15.181" xml:space="preserve">
			<polygon points="0,0 0,8.813 0.016,8.813 5.97,15.181 11.923,8.813 11.938,8.813 11.938,0 "/>
		</svg>';

		return apply_filters( 'urbango_listing_filter_get_featured_mark', $html );
	}
}
